<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Update a HubSpot company.
 */
#[Action(
  id: 'eca_hubspot_update_company',
  label: new TranslatableMarkup('HubSpot: Update Company'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Update an existing company record in HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class UpdateCompany extends HubSpotActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'company_id' => '',
      'name' => '',
      'domain' => '',
      'industry' => '',
      'phone' => '',
      'city' => '',
      'state' => '',
      'additional_properties' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['company_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company ID'),
      '#description' => $this->t('The HubSpot company ID to update.'),
      '#default_value' => $this->configuration['company_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Company Name'),
      '#description' => $this->t('The company name.'),
      '#default_value' => $this->configuration['name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['domain'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Domain'),
      '#description' => $this->t('The company website domain (e.g., example.com).'),
      '#default_value' => $this->configuration['domain'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['industry'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Industry'),
      '#description' => $this->t('The company industry.'),
      '#default_value' => $this->configuration['industry'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['phone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Phone'),
      '#description' => $this->t('The company phone number.'),
      '#default_value' => $this->configuration['phone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['city'] = [
      '#type' => 'textfield',
      '#title' => $this->t('City'),
      '#description' => $this->t('The company city.'),
      '#default_value' => $this->configuration['city'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['state'] = [
      '#type' => 'textfield',
      '#title' => $this->t('State'),
      '#description' => $this->t('The company state or region.'),
      '#default_value' => $this->configuration['state'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'company');
    $form = $this->addTokenOutputField($form, $form_state, $this->t('Provide the name of a token that holds the updated company data including ID, domain, and all properties.'));

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $company_id = $this->tokenService->replacePlain($this->configuration['company_id']);
    if (empty($company_id)) {
      $this->logError('Company ID is required to update company.');
      return;
    }

    // Build company properties to update.
    $properties = [];

    // Add properties if provided.
    $name = $this->tokenService->replaceClear($this->configuration['name']);
    if (!empty($name)) {
      $properties['name'] = $name;
    }

    $domain = $this->tokenService->replaceClear($this->configuration['domain']);
    if (!empty($domain)) {
      $properties['domain'] = $domain;
    }

    $industry = $this->tokenService->replaceClear($this->configuration['industry']);
    if (!empty($industry)) {
      $properties['industry'] = $industry;
    }

    $phone = $this->tokenService->replaceClear($this->configuration['phone']);
    if (!empty($phone)) {
      $properties['phone'] = $phone;
    }

    $city = $this->tokenService->replaceClear($this->configuration['city']);
    if (!empty($city)) {
      $properties['city'] = $city;
    }

    $state = $this->tokenService->replaceClear($this->configuration['state']);
    if (!empty($state)) {
      $properties['state'] = $state;
    }

    // Merge additional properties from YAML.
    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    // Update the company.
    $response = $this->hubspotService->updateCompany($company_id, $properties);

    // Store response in token if configured.
    $this->storeResponseToken($response);
  }

}
