<?php

declare(strict_types=1);

namespace Drupal\eca_hubspot\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Update a HubSpot ticket.
 */
#[Action(
  id: 'eca_hubspot_update_ticket',
  label: new TranslatableMarkup('HubSpot: Update Ticket'),
  category: new TranslatableMarkup('HubSpot'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Update an existing ticket in HubSpot by ID.'),
  version_introduced: '1.0.0',
)]
final class UpdateTicket extends HubSpotActionBase {

  public function defaultConfiguration(): array {
    return [
      'ticket_id' => '',
      'subject' => '',
      'content' => '',
      'pipeline' => '',
      'stage' => '',
      'priority' => '',
      'additional_properties' => '',
      'associated_contact_ids' => '',
      'associated_company_ids' => '',
      'associated_deal_ids' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['ticket_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ticket ID'),
      '#description' => $this->t('The HubSpot ticket ID to update.'),
      '#default_value' => $this->configuration['ticket_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#description' => $this->t('The ticket subject.'),
      '#default_value' => $this->configuration['subject'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Content'),
      '#description' => $this->t('The ticket content/description.'),
      '#default_value' => $this->configuration['content'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['pipeline'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Pipeline'),
      '#description' => $this->t('The pipeline ID.'),
      '#default_value' => $this->configuration['pipeline'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['stage'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Stage'),
      '#description' => $this->t('The ticket stage/status.'),
      '#default_value' => $this->configuration['stage'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['priority'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Priority'),
      '#description' => $this->t('The priority (e.g., "LOW", "MEDIUM", "HIGH").'),
      '#default_value' => $this->configuration['priority'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addYamlPropertiesField($form, $form_state, 'ticket');

    $form['associated_contact_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Contact IDs'),
      '#description' => $this->t('Comma-separated list of contact IDs.'),
      '#default_value' => $this->configuration['associated_contact_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_company_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Company IDs'),
      '#description' => $this->t('Comma-separated list of company IDs.'),
      '#default_value' => $this->configuration['associated_company_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['associated_deal_ids'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Associated Deal IDs'),
      '#description' => $this->t('Comma-separated list of deal IDs.'),
      '#default_value' => $this->configuration['associated_deal_ids'],
      '#eca_token_replacement' => TRUE,
    ];

    $form = $this->addTokenOutputField($form, $form_state, $this->t('Token that holds the updated ticket data.'));

    return $form;
  }

  public function execute(): void {
    $ticket_id = $this->tokenService->replacePlain($this->configuration['ticket_id']);
    if (empty($ticket_id)) {
      $this->logError('Ticket ID is required to update ticket.');
      return;
    }

    $properties = [];

    $subject = $this->tokenService->replaceClear($this->configuration['subject']);
    if (!empty($subject)) {
      $properties['subject'] = $subject;
    }

    $content = $this->tokenService->replace($this->configuration['content']);
    if (!empty($content)) {
      // Convert <p> and <br> to line breaks, then strip remaining HTML.
      $content = str_replace(['</p>', '<br>', '<br/>', '<br />'], "\n", $content);
      $content = strip_tags($content);
      $content = trim($content);
      $properties['content'] = $content;
    }

    $pipeline = $this->tokenService->replacePlain($this->configuration['pipeline']);
    if (!empty($pipeline)) {
      $properties['hs_pipeline'] = $pipeline;
    }

    $stage = $this->tokenService->replacePlain($this->configuration['stage']);
    if (!empty($stage)) {
      $properties['hs_pipeline_stage'] = $stage;
    }

    $priority = $this->tokenService->replacePlain($this->configuration['priority']);
    if (!empty($priority)) {
      $properties['hs_ticket_priority'] = $priority;
    }

    $properties = $this->mergeYamlSettings($properties, 'additional_properties');

    $associations = [];

    $contact_ids = $this->tokenService->replacePlain($this->configuration['associated_contact_ids']);
    if (!empty($contact_ids)) {
      $associations['contact'] = array_map('trim', explode(',', $contact_ids));
    }

    $company_ids = $this->tokenService->replacePlain($this->configuration['associated_company_ids']);
    if (!empty($company_ids)) {
      $associations['company'] = array_map('trim', explode(',', $company_ids));
    }

    $deal_ids = $this->tokenService->replacePlain($this->configuration['associated_deal_ids']);
    if (!empty($deal_ids)) {
      $associations['deal'] = array_map('trim', explode(',', $deal_ids));
    }

    $response = $this->hubspotService->updateTicket($ticket_id, $properties, $associations);
    $this->storeResponseToken($response);
  }

}
