<?php

namespace Drupal\eca_kafka\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the Kafka message template entity.
 *
 * @ConfigEntityType(
 *   id = "kafka_message_template",
 *   label = @Translation("Kafka Message Template"),
 *   config_prefix = "message_template",
 *   admin_permission = "administer site configuration",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "system_event_name",
 *     "model_id",
 *     "message_template",
 *   },
 * )
 */
class KafkaMessageTemplate extends ConfigEntityBase {

  /**
   * The machine name.
   *
   * @var string
   */
  protected $id;

  /**
   * The human-readable label.
   *
   * @var string
   */
  protected $label;

  /**
   * The system event name (e.g., "eca_render.block", "eca.content_entity.insert").
   *
   * @var string
   */
  protected $system_event_name;

  /**
   * The ECA model ID (optional for model-specific templates).
   *
   * @var string|null
   */
  protected $model_id = NULL;

  /**
   * The message template content.
   *
   * @var string
   */
  protected $message_template;

  /**
   * Gets the system event name.
   *
   * @return string
   *   The system event name (e.g., "eca_render.block").
   */
  public function getSystemEventName(): string {
    return $this->system_event_name ?? '';
  }

  /**
   * Sets the system event name.
   *
   * @param string $system_event_name
   *   The system event name.
   *
   * @return $this
   */
  public function setSystemEventName(string $system_event_name): static {
    $this->system_event_name = $system_event_name;
    return $this;
  }

  /**
   * Gets the model ID.
   *
   * @return string|null
   *   The model ID or NULL for general templates.
   */
  public function getModelId(): ?string {
    return $this->model_id;
  }

  /**
   * Sets the model ID.
   *
   * @param string|null $model_id
   *   The model ID or NULL for general templates.
   *
   * @return $this
   */
  public function setModelId(?string $model_id): static {
    $this->model_id = $model_id;
    return $this;
  }

  /**
   * Gets the message template content.
   *
   * @return string
   *   The message template content.
   */
  public function getMessageTemplate(): string {
    return $this->message_template ?? '';
  }

  /**
   * Sets the message template content.
   *
   * @param string $message_template
   *   The message template content.
   *
   * @return $this
   */
  public function setMessageTemplate(string $message_template): static {
    $this->message_template = $message_template;
    return $this;
  }

  /**
   * Gets a human-readable scope description.
   *
   * @return string
   *   The scope description.
   */
  public function getScopeDescription(): string {
    $system_event_name = $this->system_event_name ?? '';
    $model_id = $this->model_id ?? '';
    
    if ($model_id) {
      return 'Model-specific template for "' . $system_event_name . '" in model "' . $model_id . '"';
    }
    return 'General template for "' . $system_event_name . '" (applies to all models)';
  }

  /**
   * Generates a suggested machine name for a template.
   *
   * @param string $system_event_name
   *   The system event name (e.g., "eca_render.block").
   * @param string|null $model_id
   *   The model ID (optional).
   *
   * @return string
   *   The suggested machine name.
   */
  public static function generateId(string $system_event_name, ?string $model_id = NULL): string {
    // Convert system event name to safe identifier
    // "eca_render.block" -> "eca_render_block"
    // "eca.content_entity.insert" -> "eca_content_entity_insert"
    $id = str_replace(['.', ':', '-'], '_', $system_event_name);
    
    if ($model_id) {
      $id .= '__' . str_replace(['.', ':', '-'], '_', $model_id);
    }
    
    return $id;
  }

  /**
   * Gets the event plugin definition for this template's system event.
   *
   * @return array|null
   *   The event plugin definition or NULL if not found.
   */
  public function getEventPluginDefinition(): ?array {
    if (!$this->system_event_name) {
      return NULL;
    }

    $event_manager = \Drupal::service('plugin.manager.eca.event');
    $all_definitions = $event_manager->getDefinitions();

    // Find the plugin definition that matches this system event name
    foreach ($all_definitions as $plugin_id => $definition) {
      if (isset($definition['event_name']) && $definition['event_name'] === $this->system_event_name) {
        return $definition + ['plugin_id' => $plugin_id];
      }
    }

    return NULL;
  }

  /**
   * Gets the human-readable event label for this template.
   *
   * @return string
   *   The event label or the system event name if not found.
   */
  public function getEventLabel(): string {
    $definition = $this->getEventPluginDefinition();
    
    if ($definition && isset($definition['label'])) {
      return (string) $definition['label'];
    }
    
    return $this->system_event_name ?? 'Unknown Event';
  }

  /**
   * Gets the event provider (module name) for this template.
   *
   * @return string
   *   The provider module name.
   */
  public function getEventProvider(): string {
    $definition = $this->getEventPluginDefinition();
    
    if ($definition && isset($definition['provider'])) {
      return $definition['provider'];
    }
    
    return 'unknown';
  }

  // ====== LEGACY COMPATIBILITY METHODS ======
  // These methods provide backward compatibility during migration
  
  /**
   * Gets the event class name (legacy compatibility).
   *
   * @deprecated Use getSystemEventName() instead.
   * @return string
   *   The event class name or empty string.
   */
  public function getEventClassName(): string {
    // Try to extract class name from plugin definition
    $definition = $this->getEventPluginDefinition();
    
    if ($definition && isset($definition['event_class'])) {
      return basename(str_replace('\\', '/', $definition['event_class']));
    }
    
    return '';
  }

  /**
   * Gets the event plugin ID (legacy compatibility).
   *
   * @deprecated Use getSystemEventName() instead.
   * @return string
   *   The event plugin ID or empty string.
   */
  public function getEventPluginId(): string {
    $definition = $this->getEventPluginDefinition();
    
    if ($definition && isset($definition['plugin_id'])) {
      return $definition['plugin_id'];
    }
    
    return '';
  }

}
