<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_kafka\Service\KafkaProducerService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form for testing Kafka message sending functionality.
 */
class KafkaTestForm extends FormBase {

  /**
   * The Kafka producer service.
   *
   * @var \Drupal\eca_kafka\Service\KafkaProducerService
   */
  protected KafkaProducerService $kafkaProducer;

  /**
   * Constructs a new KafkaTestForm.
   *
   * @param \Drupal\eca_kafka\Service\KafkaProducerService $kafka_producer
   *   The Kafka producer service.
   */
  public function __construct(KafkaProducerService $kafka_producer) {
    $this->kafkaProducer = $kafka_producer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('eca_kafka.producer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'eca_kafka_test_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Add admin form styling following Drupal Admin Interface Patterns
    $form['#attributes']['class'][] = 'eca-kafka-test-form';
    
    $form['test_connection'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Test Connection'),
      '#attributes' => ['class' => ['eca-kafka-section']],
    ];

    $form['test_connection']['connection_test'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test Kafka Connection'),
      '#submit' => ['::testConnection'],
      '#attributes' => ['class' => ['button--primary']],
    ];

    $form['create_topic'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Create Topic'),
      '#attributes' => ['class' => ['eca-kafka-section']],
    ];

    $form['create_topic']['topic_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Topic Name'),
      '#default_value' => 'drupal_events_rest',
      '#required' => TRUE,
      '#description' => $this->t('Name of the topic to create or test with.'),
    ];

    $form['create_topic']['create_topic_btn'] = [
      '#type' => 'submit',
      '#value' => $this->t('Create Topic'),
      '#submit' => ['::createTopic'],
      '#attributes' => ['class' => ['button--secondary']],
    ];

    $form['send_message'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Send Test Message'),
      '#attributes' => ['class' => ['eca-kafka-section']],
    ];

    $form['send_message']['message_content'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message Content'),
      '#default_value' => json_encode([
        'event' => 'test_message',
        'source' => 'drupal_eca_kafka',
        'timestamp' => date('c'),
        'data' => [
          'message' => 'Hello from ECA Kafka module!',
          'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'CLI',
        ],
      ], JSON_PRETTY_PRINT),
      '#rows' => 10,
      '#required' => TRUE,
    ];

    $form['send_message']['message_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message Key (optional)'),
      '#default_value' => 'test-key-' . time(),
      '#description' => $this->t('Optional key for message partitioning.'),
    ];

    $form['send_message']['send_message_btn'] = [
      '#type' => 'submit',
      '#value' => $this->t('Send Test Message'),
      '#submit' => ['::sendMessage'],
      '#attributes' => ['class' => ['button--primary']],
    ];

    $form['get_topic_info'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Topic Information'),
      '#attributes' => ['class' => ['eca-kafka-section']],
    ];

    $form['get_topic_info']['get_info_btn'] = [
      '#type' => 'submit',
      '#value' => $this->t('Get Topic Info'),
      '#submit' => ['::getTopicInfo'],
      '#attributes' => ['class' => ['button--secondary']],
    ];

    // Attach admin CSS library following Drupal Admin Interface Patterns
    $form['#attached']['library'][] = 'eca_kafka/admin';

    return $form;
  }

  /**
   * Test the Kafka connection.
   */
  public function testConnection(array &$form, FormStateInterface $form_state) {
    try {
      $result = $this->kafkaProducer->testConnection();
      
      if ($result['success']) {
        $this->messenger()->addMessage($this->t('✅ Connection successful! Status: @status, Response time: @time ms, Cluster: @cluster', [
          '@status' => $result['status_code'],
          '@time' => $result['response_time_ms'],
          '@cluster' => $result['cluster_id'],
        ]));
      }
      else {
        $this->messenger()->addError($this->t('❌ Connection failed: @error (Status: @status)', [
          '@error' => $result['error_message'],
          '@status' => $result['status_code'],
        ]));
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('❌ Connection test error: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Create a test topic.
   */
  public function createTopic(array &$form, FormStateInterface $form_state) {
    $topic_name = $form_state->getValue('topic_name');
    
    try {
      $result = $this->kafkaProducer->createTopic($topic_name, [
        'num_partitions' => 1,
        'replication_factor' => 3,
      ]);
      
      if ($result['success']) {
        $this->messenger()->addMessage($this->t('✅ Topic "@topic" created successfully! Partitions: @partitions, Replication: @replication', [
          '@topic' => $result['topic_name'],
          '@partitions' => $result['partitions_count'],
          '@replication' => $result['replication_factor'],
        ]));
      }
      else {
        $this->messenger()->addError($this->t('❌ Failed to create topic "@topic": @error (Status: @status)', [
          '@topic' => $topic_name,
          '@error' => $result['error_message'],
          '@status' => $result['status_code'],
        ]));
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('❌ Topic creation error: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Send a test message.
   */
  public function sendMessage(array &$form, FormStateInterface $form_state) {
    $topic_name = $form_state->getValue('topic_name');
    $message_content = $form_state->getValue('message_content');
    $message_key = $form_state->getValue('message_key');
    
    try {
      $message = [
        'value' => $message_content,
      ];
      
      if (!empty($message_key)) {
        $message['key'] = $message_key;
      }
      
      $message['headers'] = [
        'X-Source' => 'drupal-eca-kafka-test',
        'X-Timestamp' => (string) time(),
        'X-Content-Type' => 'application/json',
      ];
      
      $result = $this->kafkaProducer->sendMessage($topic_name, $message);
      
      if ($result['success']) {
        $this->messenger()->addMessage($this->t('✅ Message sent successfully to topic "@topic"! Offset: @offset, Partition: @partition, Timestamp: @timestamp', [
          '@topic' => $topic_name,
          '@offset' => $result['offset'] ?? 'unknown',
          '@partition' => $result['partition'] ?? 'unknown',
          '@timestamp' => $result['timestamp'] ? date('c', $result['timestamp'] / 1000) : 'unknown',
        ]));
      }
      else {
        $this->messenger()->addError($this->t('❌ Failed to send message to topic "@topic": @error (Status: @status)', [
          '@topic' => $topic_name,
          '@error' => $result['error_message'],
          '@status' => $result['status_code'],
        ]));
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('❌ Message sending error: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * Get topic information.
   */
  public function getTopicInfo(array &$form, FormStateInterface $form_state) {
    $topic_name = $form_state->getValue('topic_name');
    
    try {
      $result = $this->kafkaProducer->getTopicInfo($topic_name);
      
      if ($result['success']) {
        if ($result['exists']) {
          $this->messenger()->addMessage($this->t('✅ Topic "@topic" exists! Partitions: @partitions, Replication: @replication', [
            '@topic' => $result['topic_name'],
            '@partitions' => $result['partitions_count'] ?? 'unknown',
            '@replication' => $result['replication_factor'] ?? 'unknown',
          ]));
        }
        else {
          $this->messenger()->addWarning($this->t('⚠️ Topic "@topic" does not exist.', [
            '@topic' => $topic_name,
          ]));
        }
      }
      else {
        $this->messenger()->addError($this->t('❌ Failed to get topic info for "@topic": @error', [
          '@topic' => $topic_name,
          '@error' => $result['error_message'],
        ]));
      }
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('❌ Topic info error: @error', [
        '@error' => $e->getMessage(),
      ]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Default submit handler - individual actions have their own handlers
  }

}
