<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_kafka\Service\EcaEventInspector;
use Drupal\eca_kafka\Service\TemplateManagerService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Html;

/**
 * Form for editing Kafka message templates with all scopes visible.
 */
class TemplateEditForm extends FormBase {

  /**
   * The ECA event inspector service.
   *
   * @var \Drupal\eca_kafka\Service\EcaEventInspector
   */
  protected EcaEventInspector $eventInspector;

  /**
   * The template manager service.
   *
   * @var \Drupal\eca_kafka\Service\TemplateManagerService
   */
  protected TemplateManagerService $templateManager;

  /**
   * Constructs a new TemplateEditForm.
   *
   * @param \Drupal\eca_kafka\Service\EcaEventInspector $event_inspector
   *   The ECA event inspector service.
   * @param \Drupal\eca_kafka\Service\TemplateManagerService $template_manager
   *   The template manager service.
   */
  public function __construct(EcaEventInspector $event_inspector, TemplateManagerService $template_manager) {
    $this->eventInspector = $event_inspector;
    $this->templateManager = $template_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('eca_kafka.event_inspector'),
      $container->get('eca_kafka.template_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eca_kafka_template_edit_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $event_class_name = NULL): array {
    if (!$event_class_name) {
      $this->messenger()->addError($this->t('Event class name is required.'));
      return $form;
    }

    // The event_class_name parameter has been cleaned for URL safety
    // We need to convert it back to the actual system event name
    $clean_event_class = urldecode($event_class_name);
    $actual_system_event_name = $this->findActualSystemEventName($clean_event_class);
    
    if (!$actual_system_event_name) {
      $this->messenger()->addError($this->t('Event class not found: @name', ['@name' => $clean_event_class]));
      return $form;
    }

    // Get the plugin ID for display purposes
    $plugin_id = $this->templateManager->getPluginIdFromSystemEventName($actual_system_event_name);
    if (!$plugin_id) {
      $this->messenger()->addError($this->t('Plugin ID not found for event class: @name', ['@name' => $actual_system_event_name]));
      return $form;
    }

    // Store the system event name for form processing
    $form_state->set('system_event_name', $actual_system_event_name);
    $form_state->set('plugin_id', $plugin_id);

    // Get event information for display
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$plugin_id] ?? [];
    $event_label = $event_definition['label'] ?? $plugin_id;

    $form['#attached']['library'][] = 'eca_kafka/template_edit';

    // Event information section
    $form['event_info'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Event Information'),
      '#attributes' => ['class' => ['event-info-section']],
    ];

    $form['event_info']['display'] = [
      '#type' => 'markup',
      '#markup' => sprintf(
        '<div class="event-details"><strong>%s</strong><br><small>Plugin ID: %s<br>System Event Name: %s</small></div>',
        htmlspecialchars($event_label),
        htmlspecialchars($plugin_id),
        htmlspecialchars($actual_system_event_name)
      ),
    ];

    // Help section with examples
    $form['help'] = [
      '#type' => 'details',
      '#title' => $this->t('Template Help & Examples'),
      '#open' => FALSE,
      '#weight' => -10,
    ];

    $form['help']['content'] = [
      '#type' => 'markup',
      '#markup' => $this->getTemplateHelp($plugin_id),
    ];

    // Get templates using TemplateManagerService
    $general_template = $this->templateManager->getTemplate($actual_system_event_name);
    
    // Get all model-specific templates for this event
    $event_usage_mapping = $this->eventInspector->getEventUsageMapping();
    $event_usage_data = $event_usage_mapping[$plugin_id] ?? null;
    
    $model_templates = [];
    if ($event_usage_data && !empty($event_usage_data['models'])) {
      foreach ($event_usage_data['models'] as $model_id => $model_info) {
        $model_template = $this->templateManager->getTemplate($actual_system_event_name, $model_id);
        if ($model_template) {
          $model_templates[$model_id] = $model_template;
        }
      }
    }

    // General Template Section
    $form['general_template'] = [
      '#type' => 'details',
      '#title' => $this->t('General Template'),
      '#description' => $this->t('This template will be used for all ECA models that trigger this event, unless overridden by a model-specific template.'),
      '#open' => TRUE,
      '#attributes' => ['class' => ['template-section']],
    ];

    $form['general_template']['general_message_template'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message Template'),
      '#default_value' => $general_template ? $general_template->getMessageTemplate() : '',
      '#rows' => 12,
      '#description' => $this->t('Enter your Kafka message template. Can be JSON, XML, plain text, or any format. You can use ECA tokens in your template.'),
      '#attributes' => [
        'class' => ['template-editor'],
      ],
    ];

    // Actions container for the general template
    $form['general_template']['actions'] = [
      '#type' => 'actions',
    ];

    $form['general_template']['actions']['save_general'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save General Template'),
      '#name' => 'save_general',
      '#submit' => ['::saveGeneralTemplate'],
      '#button_type' => 'primary',
    ];

    // Model-specific Templates Sections
    if ($event_usage_data && !empty($event_usage_data['models'])) {
      $form['model_templates_wrapper'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Model-Specific Templates'),
        '#description' => $this->t('These templates override the general template for specific ECA models.'),
        '#attributes' => ['class' => ['model-templates-wrapper']],
      ];

      foreach ($event_usage_data['models'] as $model_id => $model_info) {
        $model_label = $model_info['label'] ?? $model_id;
        $model_template = $model_templates[$model_id] ?? NULL;
        
        $field_key = 'model_' . Html::cleanCssIdentifier($model_id);
        
        // Create a details element for each model
        $form['model_templates_wrapper'][$field_key] = [
          '#type' => 'details',
          '#title' => $this->t('Model: @label', ['@label' => $model_label]),
          '#description' => $this->t('Template for the "@model" model. When set, this overrides the general template.', [
            '@model' => $model_label,
          ]),
          '#open' => !empty($model_template?->getMessageTemplate()),
          '#attributes' => ['class' => ['template-section', 'model-template-section']],
        ];

        // Use unique field names for each model's textarea
        $textarea_field_name = 'message_template_' . Html::cleanCssIdentifier($model_id);
        
        $form['model_templates_wrapper'][$field_key][$textarea_field_name] = [
          '#type' => 'textarea',
          '#title' => $this->t('Message Template'),
          '#default_value' => $model_template ? $model_template->getMessageTemplate() : '',
          '#rows' => 12,
          '#description' => $this->t('Leave empty to use the general template for this model. You can use ECA tokens in your template.'),
          '#attributes' => [
            'class' => ['template-editor'],
          ],
        ];

        // Actions container for model template
        $form['model_templates_wrapper'][$field_key]['actions'] = [
          '#type' => 'actions',
        ];

        $save_button_name = 'save_model_' . Html::cleanCssIdentifier($model_id);
        $form['model_templates_wrapper'][$field_key]['actions'][$save_button_name] = [
          '#type' => 'submit',
          '#value' => $this->t('Save @model Template', ['@model' => $model_label]),
          '#name' => $save_button_name,
          '#submit' => ['::saveModelTemplate'],
          '#button_type' => 'primary',
          '#attributes' => [
            'data-model-id' => $model_id,
            'data-textarea-field' => $textarea_field_name,
          ],
        ];

        // Add a clear button if template exists and has content
        if ($model_template && !empty($model_template->getMessageTemplate())) {
          $clear_button_name = 'clear_model_' . Html::cleanCssIdentifier($model_id);
          $form['model_templates_wrapper'][$field_key]['actions'][$clear_button_name] = [
            '#type' => 'submit',
            '#value' => $this->t('Clear Template'),
            '#name' => $clear_button_name,
            '#submit' => ['::clearModelTemplate'],
            '#button_type' => 'danger',
            '#attributes' => [
              'data-model-id' => $model_id,
            ],
          ];
        }
      }
    }

    // Back button - use standard actions container
    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['cancel'] = [
      '#type' => 'link',
      '#title' => $this->t('Back to Templates'),
      '#url' => $this->getRequest()->query->has('destination') ?
        \Drupal\Core\Url::fromUserInput($this->getRequest()->query->get('destination')) :
        \Drupal\Core\Url::fromRoute('eca_kafka.templates'),
      '#attributes' => ['class' => ['button']],
    ];

    return $form;
  }

  /**
   * Submit handler for saving the general template.
   */
  public function saveGeneralTemplate(array &$form, FormStateInterface $form_state): void {
    $system_event_name = $form_state->get('system_event_name');
    $plugin_id = $form_state->get('plugin_id');
    $message_template = $form_state->getValue('general_message_template');
    
    // Ensure message_template is never NULL - convert to empty string
    if ($message_template === NULL) {
      $message_template = '';
    }

    // Prepare template data for TemplateManagerService
    $template_data = [
      'system_event_name' => $system_event_name,
      'model_id' => NULL, // General template
      'message_template' => $message_template,
    ];

    // Get event definition for label
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$plugin_id] ?? [];
    $event_label = $event_definition['label'] ?? $plugin_id;
    $template_data['label'] = $event_label . ' Template (General)';

    // Save template using TemplateManagerService
    $result = $this->templateManager->saveTemplate($template_data);

    // Handle result
    if ($result['success']) {
      $this->messenger()->addStatus($this->t('The general template for "@event" has been saved.', [
        '@event' => $event_label,
      ]));
    } else {
      $this->messenger()->addError($result['message']);
    }

    // Stay on the same page
    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit handler for saving a model-specific template.
   */
  public function saveModelTemplate(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $model_id = $triggering_element['#attributes']['data-model-id'] ?? NULL;
    $textarea_field = $triggering_element['#attributes']['data-textarea-field'] ?? NULL;
    $system_event_name = $form_state->get('system_event_name');
    $plugin_id = $form_state->get('plugin_id');

    if (!$model_id || !$textarea_field) {
      $this->messenger()->addError($this->t('Unable to determine which template to save.'));
      return;
    }

    // Get the specific textarea value for this model
    $message_template = $form_state->getValue($textarea_field);
    
    // Ensure message_template is never NULL - convert to empty string
    if ($message_template === NULL) {
      $message_template = '';
    }

    // Prepare template data for TemplateManagerService
    $template_data = [
      'system_event_name' => $system_event_name,
      'model_id' => $model_id,
      'message_template' => $message_template,
    ];

    // Get event definition and model labels for better labeling
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$plugin_id] ?? [];
    $event_label = $event_definition['label'] ?? $plugin_id;
    
    $model_labels = $this->eventInspector->getModelLabels([$model_id]);
    $model_label = $model_labels[$model_id] ?? $model_id;
    
    $template_data['label'] = $event_label . ' Template (' . $model_label . ')';

    // Save template using TemplateManagerService
    $result = $this->templateManager->saveTemplate($template_data);

    // Handle result
    if ($result['success']) {
      $this->messenger()->addStatus($this->t('The model-specific template for "@event" in model "@model" has been saved.', [
        '@event' => $event_label,
        '@model' => $model_label,
      ]));
    } else {
      $this->messenger()->addError($result['message']);
    }

    // Stay on the same page
    $form_state->setRebuild(TRUE);
  }

  /**
   * Submit handler for clearing a model template.
   */
  public function clearModelTemplate(array &$form, FormStateInterface $form_state): void {
    $triggering_element = $form_state->getTriggeringElement();
    $model_id = $triggering_element['#attributes']['data-model-id'] ?? NULL;
    $system_event_name = $form_state->get('system_event_name');
    $plugin_id = $form_state->get('plugin_id');

    if (!$model_id) {
      $this->messenger()->addError($this->t('Unable to determine which template to clear.'));
      return;
    }

    // Get the existing template to delete it
    $template = $this->templateManager->getTemplate($system_event_name, $model_id);
    
    if ($template) {
      // Delete the template using TemplateManagerService
      $result = $this->templateManager->deleteTemplate($template->id());
      
      if ($result['success']) {
        // Get labels for success message
        $all_events = $this->eventInspector->getAllEvents();
        $event_definition = $all_events[$plugin_id] ?? [];
        $event_label = $event_definition['label'] ?? $plugin_id;
        
        $model_labels = $this->eventInspector->getModelLabels([$model_id]);
        $model_label = $model_labels[$model_id] ?? $model_id;
        
        $this->messenger()->addStatus($this->t('The model-specific template for "@event" in model "@model" has been cleared.', [
          '@event' => $event_label,
          '@model' => $model_label,
        ]));
      } else {
        $this->messenger()->addError($result['message']);
      }
    } else {
      $this->messenger()->addWarning($this->t('No template found to clear.'));
    }

    // Stay on the same page
    $form_state->setRebuild(TRUE);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // This method is not used as we have custom submit handlers
  }

  /**
   * Get template help content.
   *
   * @param string $event_class_name_param
   *   The event plugin ID.
   *
   * @return string
   *   HTML help content.
   */
  protected function getTemplateHelp(string $event_class_name_param): string {
    $help = '<h4>' . $this->t('Template Examples') . '</h4>';
    
    $help .= '<p>' . $this->t('Create templates for specific events and models to customize your Kafka messages. Templates can be JSON, XML, plain text, or any format you need.') . '</p>';
    
    $help .= '<h5>' . $this->t('JSON Message Example:') . '</h5>';
    $json_example = '{
  "event_type": "user_login",
  "timestamp": "2024-01-15T10:30:00Z",
  "source": "drupal_eca",
  "data": {
    "user_id": "[current-user:uid]",
    "user_name": "[current-user:name]",
    "email": "[current-user:mail]"
  }
}';
    $help .= '<pre>' . htmlspecialchars($json_example) . '</pre>';

    $help .= '<h5>' . $this->t('Plain Text Message Example:') . '</h5>';
    $text_example = 'User Login Event
User: [current-user:name] (ID: [current-user:uid])
Time: [date:custom:Y-m-d\TH:i:s\Z]
Email: [current-user:mail]';
    $help .= '<pre>' . htmlspecialchars($text_example) . '</pre>';

    $help .= '<h5>' . $this->t('Supported Message Formats (Kafka REST API v3)') . '</h5>';
    $help .= '<p>' . $this->t('<strong>This module uses Kafka REST API v3</strong>, which supports:') . '</p>';
    $help .= '<ul>';
    $help .= '<li>' . $this->t('<strong>JSON:</strong> Structured data (recommended for most use cases)') . '</li>';
    $help .= '<li>' . $this->t('<strong>Text/String:</strong> Plain text messages for simple notifications') . '</li>';
    $help .= '</ul>';
    $help .= '<p>' . $this->t('<strong>Note:</strong> Avro serialization requires REST API v2 and Schema Registry setup, which is not supported by this module to maintain simplicity and broad compatibility.') . '</p>';

    $help .= '<h5>' . $this->t('Using ECA Tokens:') . '</h5>';
    $help .= '<ul>';
    $help .= '<li>' . $this->t('Use ECA tokens in your templates like <code>[entity:title]</code> or <code>[current-user:uid]</code>') . '</li>';
    $help .= '<li>' . $this->t('Tokens will be replaced with actual values when the ECA model executes') . '</li>';
    $help .= '<li>' . $this->t('Templates can be any format: JSON, XML, plain text, or custom structured data') . '</li>';
    $help .= '</ul>';

    $help .= '<h5>' . $this->t('Template Hierarchy:') . '</h5>';
    $help .= '<ul>';
    $help .= '<li>' . $this->t('<strong>General Template</strong>: Applied to all models by default') . '</li>';
    $help .= '<li>' . $this->t('<strong>Model-Specific Templates</strong>: Override the general template for specific models') . '</li>';
    $help .= '<li>' . $this->t('Leave a model-specific template empty to use the general template') . '</li>';
    $help .= '</ul>';

    return $help;
  }

  /**
   * Find the actual system event name from a cleaned/URL-safe version.
   *
   * @param string $clean_event_class
   *   The cleaned event class name from the URL.
   *
   * @return string|null
   *   The actual system event name or NULL if not found.
   */
  protected function findActualSystemEventName(string $clean_event_class): ?string {
    // Get enabled events with their system event names
    $enabled_events = $this->templateManager->getEnabledEventsWithTemplatesAndModels();
    
    // First check if the clean name matches exactly (most common case)
    if (isset($enabled_events[$clean_event_class])) {
      return $clean_event_class;
    }
    
    // Search for a match by comparing cleaned versions
    foreach ($enabled_events as $system_event_name => $event_data) {
      if (Html::cleanCssIdentifier($system_event_name) === $clean_event_class) {
        return $system_event_name;
      }
    }
    
    // Also search in the enhanced templates data which uses a different structure
    // This gives us access to both plugin_id mapping and system_event_name
    $enhanced_events = $this->templateManager->getFullyEnhancedEnabledTemplates();
    foreach ($enhanced_events as $plugin_id => $event_data) {
      $system_event_name = $event_data['system_event_name'] ?? '';
      // Check both exact match and cleaned match
      if ($system_event_name === $clean_event_class || 
          Html::cleanCssIdentifier($system_event_name) === $clean_event_class) {
        return $system_event_name;
      }
    }
    
    // If still not found, maybe we received the URL parameter directly as system_event_name
    // This can happen if the class name was already valid for URLs
    return $clean_event_class;
  }

}
