<?php

namespace Drupal\eca_kafka\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\State\StateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\eca\PluginManager\Event as EcaEventManager;
use Drupal\eca_kafka\Entity\KafkaMessageTemplate;

/**
 * Service for inspecting ECA events and their usage in active models.
 */
class EcaEventInspector {

  use StringTranslationTrait;

  /**
   * The ECA event plugin manager.
   *
   * @var \Drupal\eca\PluginManager\Event
   */
  protected EcaEventManager $eventManager;

  /**
   * The state service.
   *
   * @var \Drupal\Core\State\StateInterface
   */
  protected StateInterface $state;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a new EcaEventInspector.
   *
   * @param \Drupal\eca\PluginManager\Event $event_manager
   *   The ECA event plugin manager.
   * @param \Drupal\Core\State\StateInterface $state
   *   The state service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EcaEventManager $event_manager, StateInterface $state, EntityTypeManagerInterface $entity_type_manager) {
    $this->eventManager = $event_manager;
    $this->state = $state;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Gets all available ECA event plugins.
   *
   * @return array
   *   Array of event plugin definitions keyed by plugin ID.
   */
  public function getAllEvents(): array {
    return $this->eventManager->getDefinitions();
  }

  /**
   * Gets ECA events that are actually used in active models.
   *
   * @return array
   *   Array of event plugin definitions keyed by plugin ID, filtered to only
   *   include events that are used in active ECA models.
   */
  public function getUsedEvents(): array {
    // Get events that are currently subscribed (used in active models).
    $subscribed_events = $this->state->get('eca.subscribed', []);
    $used_system_event_names = array_keys($subscribed_events);

    // Get all available event plugin definitions.
    $all_definitions = $this->getAllEvents();

    // Filter to only include events that are actually used.
    return array_filter($all_definitions, function ($definition) use ($used_system_event_names) {
      return isset($definition['event_name']) && in_array($definition['event_name'], $used_system_event_names);
    });
  }

  /**
   * Gets the ECA models (workflows) that use specific events.
   *
   * @return array
   *   Array keyed by system event name, containing arrays of models that use each event.
   *   Each model entry contains 'id', 'label', and 'entity' keys.
   */
  public function getEventModelsMapping(): array {
    $subscribed_events = $this->state->get('eca.subscribed', []);
    $event_models = [];
    
    // Get ECA entity storage to load model information
    $eca_storage = $this->entityTypeManager->getStorage('eca');
    
    // Process each subscribed event - keys are already system event names!
    foreach ($subscribed_events as $system_event_name => $priority_levels) {
      \Drupal::logger('eca_kafka')->info('Processing system event @event with @priority_count priority levels', [
        '@event' => $system_event_name,
        '@priority_count' => count($priority_levels),
      ]);
      
      if (!isset($event_models[$system_event_name])) {
        $event_models[$system_event_name] = [];
      }
      
      // Extract model IDs from all priority levels
      foreach ($priority_levels as $priority => $model_configs) {
        foreach (array_keys($model_configs) as $model_id) {
          if (!isset($event_models[$system_event_name][$model_id])) {
            // Load the ECA model entity to get human-readable information
            $eca_model = $eca_storage->load($model_id);
            if ($eca_model && $eca_model->status()) {
              $event_models[$system_event_name][$model_id] = [
                'id' => $model_id,
                'label' => $eca_model->label(),
                // Remove entity object to prevent memory issues
              ];
              \Drupal::logger('eca_kafka')->info('Added model @model (@label) to event @event', [
                '@model' => $model_id,
                '@label' => $eca_model->label(),
                '@event' => $system_event_name,
              ]);
            }
          }
        }
      }
    }
    
    \Drupal::logger('eca_kafka')->info('Final event models mapping completed with @count system events', [
      '@count' => count($event_models),
    ]);
    
    return $event_models;
  }

  /**
   * Gets human-readable labels for ECA models.
   *
   * @param array $model_ids
   *   Array of model IDs.
   *
   * @return array
   *   Array keyed by model ID with human-readable labels as values.
   */
  public function getModelLabels(array $model_ids): array {
    $labels = [];
    $eca_storage = $this->entityTypeManager->getStorage('eca');
    
    foreach ($model_ids as $model_id) {
      $eca_model = $eca_storage->load($model_id);
      if ($eca_model) {
        $labels[$model_id] = $eca_model->label();
      } else {
        $labels[$model_id] = $model_id; // Fallback to machine name
      }
    }
    
    return $labels;
  }

  /**
   * Gets events based on the filter option.
   *
   * @param bool $only_used
   *   Whether to return only events used in models (TRUE) or all events (FALSE).
   *
   * @return array
   *   Array of event plugin definitions keyed by plugin ID.
   */
  public function getEvents(bool $only_used = FALSE): array {
    return $only_used ? $this->getUsedEvents() : $this->getAllEvents();
  }

  /**
   * Gets the count of events by type.
   *
   * @return array
   *   Array with 'all' and 'used' counts.
   */
  public function getEventCounts(): array {
    return [
      'all' => count($this->getAllEvents()),
      'used' => count($this->getUsedEvents()),
    ];
  }

  /**
   * Gets the system event name from a plugin ID.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   *
   * @return string|null
   *   The system event name or NULL if not found.
   */
  public function getSystemEventNameFromPluginId(string $plugin_id): ?string {
    $all_events = $this->getAllEvents();
    $definition = $all_events[$plugin_id] ?? null;
    
    if ($definition && isset($definition['event_class'])) {
      return basename(str_replace('\\', '/', $definition['event_class']));
    }
    
    return null;
  }

  /**
   * Gets the plugin ID from a system event name.
   *
   * @param string $system_event_name
   *   The system event name.
   *
   * @return string|null
   *   The plugin ID or NULL if not found.
   */
  public function getPluginIdFromSystemEventName(string $system_event_name): ?string {
    $all_events = $this->getAllEvents();
    
    foreach ($all_events as $plugin_id => $definition) {
      if (isset($definition['event_class'])) {
        $class_name = basename(str_replace('\\', '/', $definition['event_class']));
        if ($class_name === $system_event_name) {
          return $plugin_id;
        }
      }
    }
    
    return null;
  }

  /**
   * Gets event plugin definition by system event name.
   *
   * @param string $system_event_name
   *   The system event name.
   *
   * @return array|null
   *   The event plugin definition or NULL if not found.
   */
  public function getEventDefinitionBySystemEventName(string $system_event_name): ?array {
    $plugin_id = $this->getPluginIdFromSystemEventName($system_event_name);
    
    if ($plugin_id) {
      $all_events = $this->getAllEvents();
      return $all_events[$plugin_id] ?? null;
    }
    
    return null;
  }

  /**
   * Finds the best template for an event/model combination.
   *
   * @param string $system_event_name
   *   The system event name.
   * @param string|null $model_id
   *   The model ID (optional).
   *
   * @return \Drupal\eca_kafka\Entity\KafkaMessageTemplate|null
   *   The best matching template or NULL if none found.
   */
  public function findBestTemplate(string $system_event_name, ?string $model_id = NULL): ?KafkaMessageTemplate {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');

    // First try model-specific template if model_id is provided.
    if ($model_id) {
      $specific_id = KafkaMessageTemplate::generateId($system_event_name, $model_id);
      $template = $storage->load($specific_id);
      if ($template && !empty(trim($template->getMessageTemplate()))) {
        // Only return model-specific template if it has actual content
        return $template;
      }
    }

    // Fallback to general template.
    $general_id = KafkaMessageTemplate::generateId($system_event_name);
    $general_template = $storage->load($general_id);
    
    // Return general template if it has content, otherwise return NULL
    if ($general_template && !empty(trim($general_template->getMessageTemplate()))) {
      return $general_template;
    }
    
    return NULL;
  }

  /**
   * Checks if a template exists for the given event/model combination.
   *
   * @param string $system_event_name
   *   The system event name.
   * @param string|null $model_id
   *   The model ID (optional).
   *
   * @return bool
   *   TRUE if a template exists, FALSE otherwise.
   */
  public function hasTemplate(string $system_event_name, ?string $model_id = NULL): bool {
    return $this->findBestTemplate($system_event_name, $model_id) !== NULL;
  }

  /**
   * Get currently enabled events from configuration entities keyed by plugin ID.
   *
   * This method loads all templates and converts their system event names 
   * (class names) back to plugin IDs for display purposes.
   *
   * @return array
   *   Array of enabled event configurations keyed by plugin ID.
   *   Each value contains:
   *   - plugin_id: The ECA plugin ID
   *   - template_id: The template entity ID
   *   - label: The template label
   *   - system_event_name: The class name used internally
   */
  public function getEnabledEventsByPluginId(): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadMultiple();

    $enabled = [];
    foreach ($templates as $template) {
      $system_event_name = $template->getSystemEventName();
      // Only consider general templates (no model_id) for enabled events display
      // Model-specific templates will be handled separately later
      if ($system_event_name && !$template->getModelId()) {
        // Convert system event name (class name) back to plugin ID
        $plugin_id = $this->getPluginIdFromSystemEventName($system_event_name);
        
        if ($plugin_id) {
          $enabled[$plugin_id] = [
            'plugin_id' => $plugin_id,
            'template_id' => $template->id(),
            'label' => $template->label(),
            'system_event_name' => $system_event_name,
          ];
        }
      }
    }

    return $enabled;
  }

  // ====== LEGACY COMPATIBILITY METHODS ======
  // These methods provide backward compatibility during migration

  /**
   * Gets the event class name from plugin ID (legacy compatibility).
   *
   * @deprecated Use getSystemEventNameFromPluginId() instead.
   * @param string $plugin_id
   *   The event plugin ID.
   *
   * @return string|null
   *   The event class name or NULL if not found.
   */
  public function getEventClassNameFromPluginId(string $plugin_id): ?string {
    $all_events = $this->getAllEvents();
    $definition = $all_events[$plugin_id] ?? null;
    
    if ($definition && isset($definition['event_class'])) {
      return basename(str_replace('\\', '/', $definition['event_class']));
    }
    
    return null;
  }

  /**
   * Get comprehensive event usage information for all events.
   *
   * This method provides a unified view of event usage that serves both
   * EnableEventsForm (is_used flags) and ManageTemplatesForm (detailed models).
   * It eliminates the need for separate calls to getUsedEvents() and 
   * getEventModelsMapping() and handles all key translation automatically.
   *
   * @return array
   *   Array keyed by plugin_id with comprehensive usage information:
   *   - plugin_id: The ECA event plugin ID
   *   - event_name: The event name (e.g., 'eca.content_entity.create')
   *   - system_event_name: The class name (e.g., 'ContentEntityCreate')
   *   - is_used: Boolean indicating if any active models use this event
   *   - models: Array of models using this event (with 'id' and 'label' keys)
   *   - model_count: Number of models using this event
   *   - definition: The full event plugin definition
   */
  public function getEventUsageMapping(): array {
    $all_events = $this->getAllEvents();
    $used_events = $this->getUsedEvents();  // Events with active models (keyed by plugin_id)
    $event_models_mapping = $this->getEventModelsMapping();  // Detailed models (keyed by event_name)
    
    $unified = [];
    
    foreach ($all_events as $plugin_id => $definition) {
      // Get naming schemes
      $event_name = $definition['event_name'] ?? null;
      $system_event_name = $this->getSystemEventNameFromPluginId($plugin_id);
      
      // Determine if event is used
      $is_used = isset($used_events[$plugin_id]);
      
      // Get models using this event (if any)
      $models = [];
      $model_count = 0;
      
      if ($event_name && isset($event_models_mapping[$event_name])) {
        $models = $event_models_mapping[$event_name];
        $model_count = count($models);
      }
      
      // Build unified data structure
      $unified[$plugin_id] = [
        'plugin_id' => $plugin_id,
        'event_name' => $event_name,
        'system_event_name' => $system_event_name,
        'is_used' => $is_used,
        'models' => $models,
        'model_count' => $model_count,
        'definition' => $definition,
      ];
    }
    
    return $unified;
  }

  /**
   * Gets the event name from a system event name (class name).
   *
   * This method translates from the class name format used in Kafka templates
   * (e.g., 'ContentEntityCreate') to the event name format used in ECA subscriptions
   * (e.g., 'eca.content_entity.create').
   *
   * @param string $system_event_name
   *   The system event name (class name).
   *
   * @return string|null
   *   The event name or NULL if not found.
   */
  public function getEventNameFromSystemEventName(string $system_event_name): ?string {
    // First convert system event name (class name) to plugin ID
    $plugin_id = $this->getPluginIdFromSystemEventName($system_event_name);
    
    if ($plugin_id) {
      // Then get the event_name from the plugin definition
      $all_events = $this->getAllEvents();
      $definition = $all_events[$plugin_id] ?? null;
      
      if ($definition && isset($definition['event_name'])) {
        return $definition['event_name'];
      }
    }
    
    return null;
  }

}
