<?php

namespace Drupal\eca_kafka\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Psr\Log\LoggerInterface;

/**
 * Service for producing messages to Kafka via REST API.
 */
class KafkaProducerService {

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected ClientInterface $httpClient;

  /**
   * The key repository service.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected KeyRepositoryInterface $keyRepository;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a KafkaProducerService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger channel factory.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    KeyRepositoryInterface $key_repository,
    LoggerChannelFactoryInterface $logger_factory
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->keyRepository = $key_repository;
    $this->logger = $logger_factory->get('eca_kafka');
  }

  /**
   * Sends a message to a Kafka topic.
   *
   * @param string $topic_name
   *   The name of the topic to send to.
   * @param array $message
   *   The message data to send. Should contain:
   *   - 'value': The message content (required)
   *   - 'key': Optional message key for partitioning
   *   - 'headers': Optional array of message headers
   *   - 'partition': Optional specific partition number
   *
   * @return array
   *   Response data with success status and details.
   *
   * @throws \Exception
   *   When configuration is invalid or request fails.
   */
  public function sendMessage(string $topic_name, array $message): array {
    $config = $this->configFactory->get('eca_kafka.connection');
    
    // Validate configuration
    $this->validateConfiguration($config);
    
    // Get connection details
    $kafka_rest_url = $config->get('kafka_rest_url');
    $cluster_id = $config->get('cluster_id');
    $api_credentials_key = $config->get('api_credentials');
    $timeout = $config->get('timeout') ?: 30;
    
    // Load API credentials
    $credentials = $this->getApiCredentials($api_credentials_key);
    
    // Build request URL
    $produce_url = rtrim($kafka_rest_url, '/') . '/kafka/v3/clusters/' . $cluster_id . '/topics/' . $topic_name . '/records';
    
    // Prepare message data for Kafka REST API
    $kafka_message = $this->prepareKafkaMessage($message);
    
    // Prepare authorization header
    $auth_string = base64_encode($credentials['api_key'] . ':' . $credentials['secret']);
    
    $this->logger->info('Sending message to Kafka topic @topic', [
      '@topic' => $topic_name,
      '@url' => $produce_url,
      '@message_size' => strlen(json_encode($kafka_message)),
    ]);
    
    try {
      // Make the API request
      $response = $this->httpClient->request('POST', $produce_url, [
        'headers' => [
          'Authorization' => 'Basic ' . $auth_string,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ],
        'json' => $kafka_message,
        'timeout' => $timeout,
      ]);
      
      $status_code = $response->getStatusCode();
      $response_body = $response->getBody()->getContents();
      $response_data = json_decode($response_body, TRUE);
      
      if ($status_code >= 200 && $status_code < 300) {
        $this->logger->info('Successfully sent message to Kafka topic @topic. Response: @response', [
          '@topic' => $topic_name,
          '@response' => $response_body,
        ]);
        
        return [
          'success' => TRUE,
          'status_code' => $status_code,
          'response_data' => $response_data,
          'offset' => $response_data['offset'] ?? null,
          'partition' => $response_data['partition'] ?? null,
          'timestamp' => $response_data['timestamp'] ?? null,
          'error_code' => null,
          'error_message' => null,
        ];
      }
      else {
        throw new \Exception("Unexpected status code: {$status_code}. Response: {$response_body}");
      }
      
    }
    catch (RequestException $e) {
      $error_message = $e->getMessage();
      $status_code = 0;
      $response_data = [];
      
      if ($e->hasResponse()) {
        $response = $e->getResponse();
        $status_code = $response->getStatusCode();
        $response_body = $response->getBody()->getContents();
        
        // Try to get structured error from Kafka
        $error_data = json_decode($response_body, TRUE);
        if (isset($error_data['error_code'])) {
          $error_message = $error_data['message'] ?? $error_message;
          $response_data = $error_data;
        }
      }
      
      $this->logger->error('Failed to send message to Kafka topic @topic: @error (Status: @status)', [
        '@topic' => $topic_name,
        '@error' => $error_message,
        '@status' => $status_code,
      ]);
      
      return [
        'success' => FALSE,
        'status_code' => $status_code,
        'response_data' => $response_data,
        'offset' => null,
        'partition' => null,
        'timestamp' => null,
        'error_code' => $response_data['error_code'] ?? $e->getCode(),
        'error_message' => $error_message,
      ];
    }
  }
  
  /**
   * Creates a topic in Kafka.
   *
   * @param string $topic_name
   *   The name of the topic to create.
   * @param array $config
   *   Topic configuration options:
   *   - 'num_partitions': Number of partitions (default: 1)
   *   - 'replication_factor': Replication factor (default: 3)
   *   - 'configs': Array of topic-level configurations
   *
   * @return array
   *   Response data with success status and details.
   */
  public function createTopic(string $topic_name, array $config = []): array {
    $connection_config = $this->configFactory->get('eca_kafka.connection');
    
    // Validate configuration
    $this->validateConfiguration($connection_config);
    
    // Get connection details
    $kafka_rest_url = $connection_config->get('kafka_rest_url');
    $cluster_id = $connection_config->get('cluster_id');
    $api_credentials_key = $connection_config->get('api_credentials');
    $timeout = $connection_config->get('timeout') ?: 30;
    
    // Load API credentials
    $credentials = $this->getApiCredentials($api_credentials_key);
    
    // Build request URL
    $topics_url = rtrim($kafka_rest_url, '/') . '/kafka/v3/clusters/' . $cluster_id . '/topics';
    
    // Prepare topic creation data
    $topic_data = [
      'topic_name' => $topic_name,
      'partitions_count' => $config['num_partitions'] ?? 1,
      'replication_factor' => $config['replication_factor'] ?? 3,
    ];
    
    // Add topic configurations if provided
    if (!empty($config['configs'])) {
      $topic_data['configs'] = [];
      foreach ($config['configs'] as $key => $value) {
        $topic_data['configs'][] = [
          'name' => $key,
          'value' => (string) $value,
        ];
      }
    }
    
    // Prepare authorization header
    $auth_string = base64_encode($credentials['api_key'] . ':' . $credentials['secret']);
    
    $this->logger->info('Creating Kafka topic @topic with config: @config', [
      '@topic' => $topic_name,
      '@config' => json_encode($topic_data),
    ]);
    
    try {
      // Make the API request
      $response = $this->httpClient->request('POST', $topics_url, [
        'headers' => [
          'Authorization' => 'Basic ' . $auth_string,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ],
        'json' => $topic_data,
        'timeout' => $timeout,
      ]);
      
      $status_code = $response->getStatusCode();
      $response_body = $response->getBody()->getContents();
      $response_data = json_decode($response_body, TRUE);
      
      if ($status_code >= 200 && $status_code < 300) {
        $this->logger->info('Successfully created Kafka topic @topic. Response: @response', [
          '@topic' => $topic_name,
          '@response' => $response_body,
        ]);
        
        return [
          'success' => TRUE,
          'status_code' => $status_code,
          'response_data' => $response_data,
          'topic_name' => $response_data['topic_name'] ?? $topic_name,
          'partitions_count' => $response_data['partitions_count'] ?? null,
          'replication_factor' => $response_data['replication_factor'] ?? null,
          'error_code' => null,
          'error_message' => null,
        ];
      }
      else {
        throw new \Exception("Unexpected status code: {$status_code}. Response: {$response_body}");
      }
      
    }
    catch (RequestException $e) {
      $error_message = $e->getMessage();
      $status_code = 0;
      $response_data = [];
      
      if ($e->hasResponse()) {
        $response = $e->getResponse();
        $status_code = $response->getStatusCode();
        $response_body = $response->getBody()->getContents();
        
        // Try to get structured error from Kafka
        $error_data = json_decode($response_body, TRUE);
        if (isset($error_data['error_code'])) {
          $error_message = $error_data['message'] ?? $error_message;
          $response_data = $error_data;
        }
      }
      
      $this->logger->error('Failed to create Kafka topic @topic: @error (Status: @status)', [
        '@topic' => $topic_name,
        '@error' => $error_message,
        '@status' => $status_code,
      ]);
      
      return [
        'success' => FALSE,
        'status_code' => $status_code,
        'response_data' => $response_data,
        'topic_name' => $topic_name,
        'partitions_count' => null,
        'replication_factor' => null,
        'error_code' => $response_data['error_code'] ?? $e->getCode(),
        'error_message' => $error_message,
      ];
    }
  }
  
  /**
   * Gets information about a topic.
   *
   * @param string $topic_name
   *   The name of the topic.
   *
   * @return array
   *   Topic information or error details.
   */
  public function getTopicInfo(string $topic_name): array {
    $connection_config = $this->configFactory->get('eca_kafka.connection');
    
    // Validate configuration
    $this->validateConfiguration($connection_config);
    
    // Get connection details
    $kafka_rest_url = $connection_config->get('kafka_rest_url');
    $cluster_id = $connection_config->get('cluster_id');
    $api_credentials_key = $connection_config->get('api_credentials');
    $timeout = $connection_config->get('timeout') ?: 30;
    
    // Load API credentials
    $credentials = $this->getApiCredentials($api_credentials_key);
    
    // Build request URL
    $topic_url = rtrim($kafka_rest_url, '/') . '/kafka/v3/clusters/' . $cluster_id . '/topics/' . $topic_name;
    
    // Prepare authorization header
    $auth_string = base64_encode($credentials['api_key'] . ':' . $credentials['secret']);
    
    try {
      // Make the API request
      $response = $this->httpClient->request('GET', $topic_url, [
        'headers' => [
          'Authorization' => 'Basic ' . $auth_string,
          'Accept' => 'application/json',
        ],
        'timeout' => $timeout,
      ]);
      
      $status_code = $response->getStatusCode();
      $response_body = $response->getBody()->getContents();
      $response_data = json_decode($response_body, TRUE);
      
      if ($status_code >= 200 && $status_code < 300) {
        return [
          'success' => TRUE,
          'exists' => TRUE,
          'topic_data' => $response_data,
          'topic_name' => $response_data['topic_name'] ?? $topic_name,
          'partitions_count' => $response_data['partitions_count'] ?? null,
          'replication_factor' => $response_data['replication_factor'] ?? null,
          'error_code' => null,
          'error_message' => null,
        ];
      }
      else {
        throw new \Exception("Unexpected status code: {$status_code}. Response: {$response_body}");
      }
      
    }
    catch (RequestException $e) {
      $status_code = $e->hasResponse() ? $e->getResponse()->getStatusCode() : 0;
      
      // Topic not found is a common case
      if ($status_code === 404) {
        return [
          'success' => TRUE,
          'exists' => FALSE,
          'topic_data' => null,
          'topic_name' => $topic_name,
          'partitions_count' => null,
          'replication_factor' => null,
          'error_code' => 404,
          'error_message' => 'Topic not found',
        ];
      }
      
      $error_message = $e->getMessage();
      
      $this->logger->error('Failed to get Kafka topic info for @topic: @error (Status: @status)', [
        '@topic' => $topic_name,
        '@error' => $error_message,
        '@status' => $status_code,
      ]);
      
      return [
        'success' => FALSE,
        'exists' => NULL,
        'topic_data' => null,
        'topic_name' => $topic_name,
        'partitions_count' => null,
        'replication_factor' => null,
        'error_code' => $e->getCode(),
        'error_message' => $error_message,
      ];
    }
  }
  
  /**
   * Tests the connection to Kafka REST API.
   *
   * @return array
   *   Connection test results.
   */
  public function testConnection(): array {
    $config = $this->configFactory->get('eca_kafka.connection');
    
    try {
      // Validate configuration
      $this->validateConfiguration($config);
      
      // Get connection details
      $kafka_rest_url = $config->get('kafka_rest_url');
      $cluster_id = $config->get('cluster_id');
      $api_credentials_key = $config->get('api_credentials');
      $timeout = $config->get('timeout') ?: 30;
      
      // Load API credentials
      $credentials = $this->getApiCredentials($api_credentials_key);
      
      // Test by getting cluster info
      $cluster_url = rtrim($kafka_rest_url, '/') . '/kafka/v3/clusters/' . $cluster_id;
      $auth_string = base64_encode($credentials['api_key'] . ':' . $credentials['secret']);
      
      $start_time = microtime(TRUE);
      
      $response = $this->httpClient->request('GET', $cluster_url, [
        'headers' => [
          'Authorization' => 'Basic ' . $auth_string,
          'Accept' => 'application/json',
        ],
        'timeout' => $timeout,
      ]);
      
      $response_time = round((microtime(TRUE) - $start_time) * 1000, 2);
      $status_code = $response->getStatusCode();
      
      if ($status_code >= 200 && $status_code < 300) {
        return [
          'success' => TRUE,
          'status_code' => $status_code,
          'response_time_ms' => $response_time,
          'cluster_id' => $cluster_id,
          'error_message' => null,
        ];
      }
      else {
        return [
          'success' => FALSE,
          'status_code' => $status_code,
          'response_time_ms' => $response_time,
          'cluster_id' => $cluster_id,
          'error_message' => "Unexpected status code: {$status_code}",
        ];
      }
      
    }
    catch (RequestException $e) {
      return [
        'success' => FALSE,
        'status_code' => $e->hasResponse() ? $e->getResponse()->getStatusCode() : 0,
        'response_time_ms' => null,
        'cluster_id' => $config->get('cluster_id'),
        'error_message' => $e->getMessage(),
      ];
    }
    catch (\Exception $e) {
      return [
        'success' => FALSE,
        'status_code' => 0,
        'response_time_ms' => null,
        'cluster_id' => $config->get('cluster_id'),
        'error_message' => $e->getMessage(),
      ];
    }
  }
  
  /**
   * Validates the Kafka connection configuration.
   *
   * @param \Drupal\Core\Config\Config $config
   *   The connection configuration.
   *
   * @throws \Exception
   *   When configuration is invalid.
   */
  protected function validateConfiguration($config): void {
    $kafka_rest_url = $config->get('kafka_rest_url');
    $cluster_id = $config->get('cluster_id');
    $api_credentials_key = $config->get('api_credentials');
    
    if (empty($kafka_rest_url)) {
      throw new \Exception('Kafka REST URL is not configured');
    }
    
    if (empty($cluster_id)) {
      throw new \Exception('Kafka cluster ID is not configured');
    }
    
    if (empty($api_credentials_key)) {
      throw new \Exception('API credentials are not configured');
    }
  }
  
  /**
   * Gets API credentials from the key repository.
   *
   * @param string $api_credentials_key
   *   The key ID for the API credentials.
   *
   * @return array
   *   Array with 'api_key' and 'secret' keys.
   *
   * @throws \Exception
   *   When credentials cannot be loaded.
   */
  protected function getApiCredentials(string $api_credentials_key): array {
    $key = $this->keyRepository->getKey($api_credentials_key);
    if (!$key) {
      throw new \Exception("API credentials key '{$api_credentials_key}' not found");
    }
    
    $credentials = $key->getKeyValues();
    
    // Try different common field name combinations
    $api_key = null;
    $secret = null;
    
    // Check for api_key/secret combination
    if (isset($credentials['api_key']) && isset($credentials['secret'])) {
      $api_key = trim($credentials['api_key']);
      $secret = trim($credentials['secret']);
    }
    // Check for username/password combination (common default)
    elseif (isset($credentials['username']) && isset($credentials['password'])) {
      $api_key = trim($credentials['username']);
      $secret = trim($credentials['password']);
    }
    // Check for key/secret combination
    elseif (isset($credentials['key']) && isset($credentials['secret'])) {
      $api_key = trim($credentials['key']);
      $secret = trim($credentials['secret']);
    }
    
    if (empty($api_key) || empty($secret)) {
      $available_fields = implode(', ', array_keys($credentials));
      throw new \Exception("API credentials not found in key. Available fields: {$available_fields}. Expected one of: api_key/secret, username/password, or key/secret combinations.");
    }
    
    return [
      'api_key' => $api_key,
      'secret' => $secret,
    ];
  }
  
  /**
   * Prepares message data for Kafka REST API format.
   *
   * @param array $message
   *   The message data from the caller.
   *
   * @return array
   *   Formatted message data for Kafka REST API.
   */
  protected function prepareKafkaMessage(array $message): array {
    $kafka_message = [];
    
    // Value is required
    if (!isset($message['value'])) {
      throw new \InvalidArgumentException('Message value is required');
    }
    
    // Prepare value in Confluent Cloud REST API v3 format
    if (is_array($message['value']) || is_object($message['value'])) {
      // For complex data, use JSON type with JSON-encoded data
      $kafka_message['value'] = [
        'type' => 'JSON',
        'data' => json_encode($message['value']),
      ];
    }
    else {
      // For simple strings, use STRING type
      $kafka_message['value'] = [
        'type' => 'STRING',
        'data' => (string) $message['value'],
      ];
    }
    
    // Optional key for partitioning (always STRING type)
    if (isset($message['key'])) {
      $kafka_message['key'] = [
        'type' => 'STRING',
        'data' => (string) $message['key'],
      ];
    }
    
    // Optional headers (must be base64 encoded in Confluent format)
    if (isset($message['headers']) && is_array($message['headers'])) {
      $kafka_message['headers'] = [];
      foreach ($message['headers'] as $header_key => $header_value) {
        $kafka_message['headers'][] = [
          'name' => $header_key,
          'value' => base64_encode((string) $header_value),
        ];
      }
    }
    
    // Optional specific partition
    if (isset($message['partition'])) {
      $kafka_message['partition_id'] = (int) $message['partition'];
    }
    
    return $kafka_message;
  }

  /**
   * Lists all topics in the Kafka cluster.
   *
   * @return array
   *   Response data with success status and topics list.
   *   On success, contains:
   *   - 'success': TRUE
   *   - 'topics': Array of topic names
   *   - 'total_count': Total number of topics
   *   On failure, contains:
   *   - 'success': FALSE
   *   - 'error_message': Error description
   *   - 'status_code': HTTP status code
   */
  public function listTopics(): array {
    $config = $this->configFactory->get('eca_kafka.connection');
    
    try {
      // Validate configuration
      $this->validateConfiguration($config);
      
      // Get connection details
      $kafka_rest_url = $config->get('kafka_rest_url');
      $cluster_id = $config->get('cluster_id');
      $api_credentials_key = $config->get('api_credentials');
      $timeout = $config->get('timeout') ?: 30;
      
      // Load API credentials
      $credentials = $this->getApiCredentials($api_credentials_key);
      
      // Build request URL for listing topics
      $topics_url = rtrim($kafka_rest_url, '/') . '/kafka/v3/clusters/' . $cluster_id . '/topics';
      
      // Prepare authorization header
      $auth_string = base64_encode($credentials['api_key'] . ':' . $credentials['secret']);
      
      $this->logger->info('Fetching Kafka topics list from cluster @cluster', [
        '@cluster' => $cluster_id,
      ]);
      
      // Make the HTTP request
      $response = $this->httpClient->request('GET', $topics_url, [
        'timeout' => $timeout,
        'headers' => [
          'Authorization' => 'Basic ' . $auth_string,
          'Content-Type' => 'application/json',
          'Accept' => 'application/json',
        ],
      ]);
      
      $status_code = $response->getStatusCode();
      $response_body = $response->getBody()->getContents();
      
      if ($status_code === 200) {
        $data = json_decode($response_body, TRUE);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
          throw new \RuntimeException('Invalid JSON response from Kafka API');
        }
        
        // Extract topic names from the response
        $topics = [];
        if (isset($data['data']) && is_array($data['data'])) {
          foreach ($data['data'] as $topic) {
            if (isset($topic['topic_name'])) {
              $topics[] = $topic['topic_name'];
            }
          }
        }
        
        $this->logger->info('Successfully fetched @count Kafka topics', [
          '@count' => count($topics),
        ]);
        
        return [
          'success' => TRUE,
          'topics' => $topics,
          'total_count' => count($topics),
          'status_code' => $status_code,
        ];
      }
      else {
        $this->logger->error('Failed to list Kafka topics. Status: @status, Response: @response', [
          '@status' => $status_code,
          '@response' => $response_body,
        ]);
        
        return [
          'success' => FALSE,
          'topics' => [],
          'total_count' => 0,
          'error_message' => "HTTP {$status_code}: Failed to fetch topics list",
          'status_code' => $status_code,
        ];
      }
      
    } catch (RequestException $e) {
      $this->logger->error('HTTP error while listing Kafka topics: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      return [
        'success' => FALSE,
        'topics' => [],
        'total_count' => 0,
        'error_message' => 'Connection error: ' . $e->getMessage(),
        'status_code' => $e->getCode(),
      ];
    } catch (\Exception $e) {
      $this->logger->error('Unexpected error while listing Kafka topics: @error', [
        '@error' => $e->getMessage(),
      ]);
      
      return [
        'success' => FALSE,
        'topics' => [],
        'total_count' => 0,
        'error_message' => $e->getMessage(),
        'status_code' => 0,
      ];
    }
  }

}
