(function ($, Drupal, drupalSettings) {
  'use strict';

  Drupal.behaviors.ecaKafkaEventSelector = {
    attach: function (context, settings) {
      // Only attach once per page load
      if ($(context).hasClass('eca-kafka-selector-processed')) {
        return;
      }
      $(context).addClass('eca-kafka-selector-processed');

      // Get event data from Drupal settings
      var allEvents = settings.eca_kafka?.all_events || [];
      var selectedEvents = [];
      
      var $searchInput = $('#event-search-input', context);
      var $searchResults = $('#search-results', context);
      var $selectedList = $('#selected-events .selected-list', context);
      var $hiddenField = $('#selected-plugin-ids', context);
      var $submitButton = $('#enable-events-submit', context);

      // Skip if elements not found (prevents errors on other pages)
      if ($searchInput.length === 0) {
        return;
      }

      // Initialize
      updateSubmitButton();

      // Search functionality
      $searchInput.on('keyup', function() {
        var query = $(this).val().toLowerCase().trim();
        
        if (query.length < 2) {
          $searchResults.empty();
          return;
        }

        var filteredEvents = allEvents.filter(function(event) {
          return event.search_text.indexOf(query) !== -1;
        });

        displaySearchResults(filteredEvents);
      });

      function displaySearchResults(events) {
        $searchResults.empty();
        
        if (events.length === 0) {
          $searchResults.html('<div class="no-results">No events found matching your search.</div>');
          return;
        }

        // Limit to first 10 results
        var displayEvents = events.slice(0, 10);
        
        displayEvents.forEach(function(event) {
          // Skip if already selected
          if (selectedEvents.indexOf(event.plugin_id) !== -1) {
            return;
          }

          var statusBadge = event.is_used 
            ? '<span class="badge badge-used">Used in models</span>'
            : '<span class="badge badge-unused">Not used</span>';

          var eventItem = $('<div class="search-result-item" data-plugin-id="' + event.plugin_id + '">' +
            '<div class="event-info">' +
              '<strong>' + event.label + '</strong> ' + statusBadge + '<br>' +
              '<small>Plugin: ' + event.plugin_id + ' | Provider: ' + event.provider + '</small>' +
            '</div>' +
            '<button type="button" class="add-event-btn">Add</button>' +
          '</div>');

          $searchResults.append(eventItem);
        });

        if (events.length > 10) {
          $searchResults.append('<div class="more-results">Showing first 10 results. Continue typing to narrow search.</div>');
        }
      }

      // Handle adding events
      $searchResults.on('click', '.add-event-btn', function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        var $button = $(this);
        var $item = $button.closest('.search-result-item');
        var pluginId = $item.data('plugin-id');
        
        // Prevent double clicks
        if ($button.prop('disabled')) {
          return;
        }
        $button.prop('disabled', true).text('Adding...');
        
        // Find the event data
        var eventData = allEvents.find(function(event) {
          return event.plugin_id === pluginId;
        });
        
        // Double-check not already selected
        if (eventData && selectedEvents.indexOf(pluginId) === -1) {
          selectedEvents.push(pluginId);
          addToSelectedList(eventData);
          updateHiddenField();
          updateSubmitButton();
          
          // Remove from search results
          $item.fadeOut(function() {
            $(this).remove();
          });

          // Clear search if no more results
          if ($searchResults.find('.search-result-item').length === 1) {
            $searchInput.val('');
            $searchResults.empty();
          }
        } else {
          // Re-enable button if something went wrong
          $button.prop('disabled', false).text('Add');
        }
      });

      // Handle removing events from selected list
      $selectedList.on('click', '.remove-event-btn', function(e) {
        e.preventDefault();
        var pluginId = $(this).data('plugin-id');
        
        // Remove from selected array
        var index = selectedEvents.indexOf(pluginId);
        if (index > -1) {
          selectedEvents.splice(index, 1);
        }
        
        // Remove from UI
        $(this).closest('.selected-event-item').fadeOut(function() {
          $(this).remove();
          updateSelectedSection();
        });
        
        updateHiddenField();
        updateSubmitButton();
      });

      // Handle disabling enabled events using the disable button
      $(document).on('click', '.disable-event-btn', function(e) {
        e.preventDefault();
        var $link = $(this);
        var pluginId = $link.data('plugin-id');
        var eventLabel = $link.data('event-label');
        
        // Show confirmation dialog
        var confirmMessage = Drupal.t('Are you sure you want to disable "@event"?\n\nThis will remove ALL templates (general and model-specific) for this event.', {
          '@event': eventLabel
        });
        
        if (confirm(confirmMessage)) {
          // Set the disable plugin ID in hidden field
          $('#disable-plugin-id').val(pluginId);
          
          // Trigger form submit using jQuery to work with Drupal form handling
          var $form = $('form#eca-kafka-enable-events-form');
          if ($form.length > 0) {
            // Use native form submit to avoid AJAX conflicts
            $form[0].submit();
          }
        }
      });

      function addToSelectedList(eventData) {
        var statusBadge = eventData.is_used 
          ? '<span class="badge badge-used">Used in models</span>'
          : '<span class="badge badge-unused">Not used</span>';

        var selectedItem = $('<div class="selected-event-item" data-plugin-id="' + eventData.plugin_id + '">' +
          '<div class="event-info">' +
            '<strong>' + eventData.label + '</strong> ' + statusBadge + '<br>' +
            '<small>Plugin: ' + eventData.plugin_id + ' | Provider: ' + eventData.provider + '</small>' +
          '</div>' +
          '<button type="button" class="remove-event-btn" data-plugin-id="' + eventData.plugin_id + '">Remove</button>' +
        '</div>');

        $selectedList.append(selectedItem);
        updateSelectedSection();
      }

      function updateSelectedSection() {
        var $container = $('#selected-events');
        if (selectedEvents.length > 0) {
          $container.show();
        } else {
          $container.hide();
        }
      }

      function updateHiddenField() {
        $hiddenField.val(selectedEvents.join(','));
      }

      function updateSubmitButton() {
        if (selectedEvents.length > 0) {
          $submitButton.prop('disabled', false).text(
            Drupal.formatPlural(selectedEvents.length, 'Enable 1 Event', 'Enable @count Events')
          );
        } else {
          $submitButton.prop('disabled', true).text('Enable Selected Events');
        }
      }

      // Initialize selected section visibility
      updateSelectedSection();
    }
  };

})(jQuery, Drupal, drupalSettings);
