<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Kafka connection settings.
 */
class ConnectionConfigForm extends ConfigFormBase {

  /**
   * Constructs a ConnectionConfigForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager
  ) {
    parent::__construct($config_factory, $typed_config_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['eca_kafka.connection'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'eca_kafka_connection_config';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('eca_kafka.connection');

    // Add admin form styling
    $form['#attributes']['class'][] = 'eca-kafka-admin-form';
    
    $form['kafka_rest_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Kafka REST Proxy URL'),
      '#description' => $this->t('The URL of your Kafka REST Proxy server. Examples:<br>
        • Confluent Cloud: <code>https://pkc-xxxxx.region.provider.confluent.cloud:443</code><br>
        • Self-hosted Confluent: <code>http://localhost:8082</code><br>
        • Karapace: <code>http://localhost:8082</code>'),
      '#default_value' => $config->get('kafka_rest_url') ?: '',
      '#required' => TRUE,
      '#placeholder' => 'https://pkc-xxxxx.region.provider.confluent.cloud:443',
    ];

    $form['cluster_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Cluster ID'),
      '#description' => $this->t('The ID of your Kafka cluster (e.g., lkc-xxxxx for Confluent Cloud).'),
      '#default_value' => $config->get('cluster_id') ?: '',
      '#required' => TRUE,
      '#placeholder' => 'lkc-xxxxx',
    ];

    $form['api_credentials'] = [
      '#type' => 'key_select',
      '#title' => $this->t('API Credentials'),
      '#description' => $this->t('Select a key containing your Kafka API key and secret. The key should be of type "Authentication (Multivalue)" with fields:<br>
        • <strong>api_key</strong>: Your Kafka API key<br>
        • <strong>secret</strong>: Your Kafka API secret<br>
        Or use <strong>username</strong>/<strong>password</strong> fields.'),
      '#default_value' => $config->get('api_credentials') ?: '',
      '#key_filters' => [
        'type' => 'authentication_multivalue',
      ],
      '#required' => TRUE,
    ];

    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced settings'),
      '#open' => FALSE,
      '#attributes' => ['class' => ['eca-kafka-advanced-settings']],
    ];

    $form['advanced']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request timeout'),
      '#description' => $this->t('HTTP request timeout in seconds.'),
      '#default_value' => $config->get('timeout') ?: 30,
      '#min' => 1,
      '#max' => 300,
      '#required' => TRUE,
    ];

    // Information message following Drupal patterns
    $form['testing_info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="messages messages--info eca-kafka-info">' . 
        $this->t('After saving your configuration, use the <strong>Test Kafka</strong> tab to test your connection and send messages.') .
        '</div>',
      '#weight' => 100,
    ];

    // Attach admin CSS library
    $form['#attached']['library'][] = 'eca_kafka/admin';

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    // Validate URL format
    $url = $form_state->getValue('kafka_rest_url');
    if (!filter_var($url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('kafka_rest_url', $this->t('Please enter a valid URL.'));
    }

    // Ensure URL doesn't end with trailing slash
    if (substr($url, -1) === '/') {
      $form_state->setValue('kafka_rest_url', rtrim($url, '/'));
    }

    // Validate cluster ID format (basic check)
    $cluster_id = $form_state->getValue('cluster_id');
    if (!preg_match('/^[a-zA-Z0-9_-]+$/', $cluster_id)) {
      $form_state->setErrorByName('cluster_id', $this->t('Cluster ID should contain only letters, numbers, hyphens, and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('eca_kafka.connection')
      ->set('kafka_rest_url', $form_state->getValue('kafka_rest_url'))
      ->set('cluster_id', $form_state->getValue('cluster_id'))
      ->set('api_credentials', $form_state->getValue('api_credentials'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->save();

    $this->messenger()->addMessage($this->t('Kafka connection configuration has been saved.'));

    parent::submitForm($form, $form_state);
  }

}
