<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\eca_kafka\Service\EcaEventInspector;
use Drupal\eca_kafka\Service\TemplateManagerService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Utility\Html;

/**
 * Form for enabling ECA events for Kafka publishing.
 */
class EnableEventsForm extends FormBase {

  /**
   * The ECA event inspector service.
   *
   * @var \Drupal\eca_kafka\Service\EcaEventInspector
   */
  protected EcaEventInspector $eventInspector;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The template manager service.
   *
   * @var \Drupal\eca_kafka\Service\TemplateManagerService
   */
  protected TemplateManagerService $templateManager;

  /**
   * Constructs a new EnableEventsForm.
   *
   * @param \Drupal\eca_kafka\Service\EcaEventInspector $event_inspector
   *   The ECA event inspector service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\eca_kafka\Service\TemplateManagerService $template_manager
   *   The template manager service.
   */
  public function __construct(EcaEventInspector $event_inspector, EntityTypeManagerInterface $entity_type_manager, TemplateManagerService $template_manager) {
    $this->eventInspector = $event_inspector;
    $this->entityTypeManager = $entity_type_manager;
    $this->templateManager = $template_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('eca_kafka.event_inspector'),
      $container->get('entity_type.manager'),
      $container->get('eca_kafka.template_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eca_kafka_enable_events_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['#attached']['library'][] = 'eca_kafka/event_selector';

    // Description
    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Enable or disable ECA events for Kafka message publishing. Currently enabled events are shown below with operations. Search for new events and select them to enable.') . '</p>',
    ];

    // Use centralized enhanced template service for consistent data
    $enabled_events = $this->templateManager->getEnhancedEnabledTemplates();
    
    $form['current_events_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Event'),
        $this->t('Status'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No events are currently enabled for Kafka publishing.'),
      '#attributes' => ['class' => ['enabled-events-table']],
    ];

    foreach ($enabled_events as $plugin_id => $event_data) {
      $form['current_events_table'][$plugin_id] = $this->buildEnabledEventRow($plugin_id, $event_data);
    }

    // Event search and selection section
    $form['add_events'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Add Events'),
      '#attributes' => ['class' => ['add-events-section']],
    ];

    $form['add_events']['search'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Search Events'),
      '#placeholder' => $this->t('Type event name or plugin ID...'),
      '#attributes' => [
        'id' => 'event-search-input',
        'autocomplete' => 'off',
      ],
      '#description' => $this->t('Start typing to search for available events. Click on an event to add it to your selection.'),
    ];

    // Search results container (populated by JavaScript)
    $form['add_events']['search_results'] = [
      '#type' => 'markup',
      '#markup' => '<div id="search-results" class="search-results-container"></div>',
    ];

    // Selected events to enable (populated by JavaScript)
    $form['add_events']['selected_events'] = [
      '#type' => 'markup',
      '#markup' => '<div id="selected-events" class="selected-events-container"><h4>' . $this->t('Selected to Enable:') . '</h4><div class="selected-list"></div></div>',
    ];

    // Hidden field to store selected event plugin IDs for enabling
    $form['selected_plugin_ids'] = [
      '#type' => 'hidden',
      '#attributes' => ['id' => 'selected-plugin-ids'],
    ];

    // Hidden field to store event plugin ID to disable
    $form['disable_plugin_id'] = [
      '#type' => 'hidden',
      '#attributes' => ['id' => 'disable-plugin-id'],
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Enable Selected Events'),
      '#button_type' => 'primary',
      '#attributes' => ['id' => 'enable-events-submit'],
    ];

    // Provide event data for JavaScript
    $form['#attached']['drupalSettings']['eca_kafka'] = [
      'all_events' => $this->getAllEventsForJs(),
      'enabled_events' => array_keys($enabled_events),
    ];

    return $form;
  }

  /**
   * Build a table row for an enabled event.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   * @param array $event_data
   *   The enhanced event data from templateManager->getEnhancedEnabledTemplates().
   *
   * @return array
   *   The form row array.
   */
  protected function buildEnabledEventRow(string $plugin_id, array $event_data): array {
    // Use centralized formatting service
    $display_data = $this->templateManager->formatEventDisplayData($plugin_id, $event_data);

    // Event column - show class name prominently, plugin ID for context
    $row['event'] = [
      '#type' => 'markup',
      '#markup' => sprintf(
        '<div class="event-info"><strong>%s</strong><br><small>%s | %s</small></div>',
        Html::escape($display_data['system_event_name']),
        Html::escape($display_data['plugin_id']),
        Html::escape($display_data['provider'])
      ),
    ];

    // Status column
    $row['status'] = [
      '#type' => 'markup',
      '#markup' => '<span class="status-badge status-enabled">' . $this->t('Enabled') . '</span>',
    ];

    // Operations column - HTML button approach
    $row['operations'] = [
      '#type' => 'inline_template',
      '#template' => '<button type="button" class="disable-event-btn button button--small" data-plugin-id="{{ plugin_id }}" data-event-label="{{ label }}">{{ disable_text }}</button>',
      '#context' => [
        'plugin_id' => $plugin_id,
        'label' => $display_data['display_label'],
        'disable_text' => $this->t('Disable'),
      ],
    ];

    return $row;
  }

  /**
   * Get all events formatted for JavaScript use.
   *
   * @return array
   *   Array of event data for JavaScript.
   */
  protected function getAllEventsForJs(): array {
    // Use centralized service method
    return $this->templateManager->getAvailableEventsForJs();
  }

  /**

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Handle disable operation if disable_plugin_id is set
    $disable_plugin_id = $form_state->getValue('disable_plugin_id', '');
    if (!empty($disable_plugin_id)) {
      $this->disableEvent($disable_plugin_id);
      // Redirect to prevent resubmission
      $form_state->setRedirect('eca_kafka.enable_events');
      return;
    }

    // Handle enable operation
    $selected_plugin_ids = $form_state->getValue('selected_plugin_ids', '');
    $plugin_ids = array_filter(explode(',', $selected_plugin_ids));

    if (empty($plugin_ids)) {
      $this->messenger()->addWarning($this->t('No events were selected to enable.'));
      return;
    }

    // Use centralized bulk enable operation
    $result = $this->templateManager->enableMultipleEvents($plugin_ids);

    // Handle results using centralized response data
    if ($result['success_count'] > 0) {
      $this->messenger()->addStatus($this->t('Successfully enabled @count events for Kafka publishing.', [
        '@count' => $result['success_count'],
      ]));

      $this->messenger()->addStatus($this->t('You can now configure message templates for these events in the <a href="@url">Manage Templates</a> section.', [
        '@url' => '/admin/config/workflow/eca/kafka/templates',
      ]));
    }
    
    // Show skipped events as warnings
    foreach ($result['skipped_messages'] as $message) {
      $this->messenger()->addWarning($message);
    }
    
    // Show errors
    foreach ($result['error_messages'] as $message) {
      $this->messenger()->addError($message);
    }
    
    // Final summary if nothing was created
    if ($result['success_count'] === 0 && empty($result['error_messages'])) {
      $this->messenger()->addWarning($this->t('No new events were created. Check the messages above for details.'));
    }
  }

  /**
   * Disable an event using the centralized template manager service.
   *
   * @param string $plugin_id
   *   The event plugin ID to disable.
   */
  protected function disableEvent(string $plugin_id): void {
    // Use centralized template service instead of manual CRUD
    $result = $this->templateManager->disableEvent($plugin_id);
    
    if ($result['success']) {
      $this->messenger()->addStatus($result['message']);
    } else {
      $this->messenger()->addError($result['message']);
    }
  }

}
