<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\eca_kafka\Service\EcaEventInspector;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form that displays all ECA events.
 */
class EventListForm extends FormBase {

  /**
   * The ECA event inspector service.
   *
   * @var \Drupal\eca_kafka\Service\EcaEventInspector
   */
  protected EcaEventInspector $eventInspector;

  /**
   * Constructs a new EventListForm.
   *
   * @param \Drupal\eca_kafka\Service\EcaEventInspector $event_inspector
   *   The ECA event inspector service.
   */
  public function __construct(EcaEventInspector $event_inspector) {
    $this->eventInspector = $event_inspector;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('eca_kafka.event_inspector')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eca_kafka_event_list_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $counts = $this->eventInspector->getEventCounts();
    
    $form['info'] = [
      '#type' => 'markup',
      '#markup' => '<p>Total events: <strong>' . $counts['all'] . '</strong> | Used in models: <strong>' . $counts['used'] . '</strong></p>',
    ];

    // Get unified event usage data
    $event_usage_mapping = $this->eventInspector->getEventUsageMapping();
    
    $rows = [];
    foreach ($event_usage_mapping as $plugin_id => $event_data) {
      $definition = $event_data['definition'];
      $label = $definition['label'] ?? $plugin_id;
      $event_name = $event_data['event_name'] ?? 'N/A';
      $provider = $definition['provider'] ?? 'Unknown';
      $is_used = $event_data['is_used'] ? 'Yes' : 'No';
      
      $rows[] = [
        'plugin_id' => $plugin_id,
        'label' => $label,
        'event_name' => $event_name,
        'provider' => $provider,
        'used' => $is_used,
      ];
    }

    $form['events_table'] = [
      '#type' => 'table',
      '#header' => [
        'plugin_id' => $this->t('Plugin ID'),
        'label' => $this->t('Label'),
        'event_name' => $this->t('Event Name'),
        'provider' => $this->t('Provided By'),
        'used' => $this->t('Used in Models'),
      ],
      '#rows' => $rows,
      '#empty' => $this->t('No events found.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // No submit needed - this is just a display form.
  }

}
