<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\eca_kafka\Service\TemplateManagerService;
use Drupal\eca_kafka\Service\EcaEventInspector;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\Component\Utility\Html;


/**
 * Form for managing Kafka message templates.
 */
class ManageTemplatesForm extends FormBase {

  /**
   * The template manager service.
   *
   * @var \Drupal\eca_kafka\Service\TemplateManagerService
   */
  protected TemplateManagerService $templateManager;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The ECA event inspector service.
   *
   * @var \Drupal\eca_kafka\Service\EcaEventInspector
   */
  protected EcaEventInspector $eventInspector;

  /**
   * Constructs a new ManageTemplatesForm.
   *
   * @param \Drupal\eca_kafka\Service\TemplateManagerService $template_manager
   *   The template manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\eca_kafka\Service\EcaEventInspector $event_inspector
   *   The ECA event inspector service.
   */
  public function __construct(TemplateManagerService $template_manager, EntityTypeManagerInterface $entity_type_manager, EcaEventInspector $event_inspector) {
    $this->templateManager = $template_manager;
    $this->entityTypeManager = $entity_type_manager;
    $this->eventInspector = $event_inspector;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('eca_kafka.template_manager'),
      $container->get('entity_type.manager'),
      $container->get('eca_kafka.event_inspector')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eca_kafka_manage_templates_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['#attached']['library'][] = 'eca_kafka/manage_templates';

    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Manage message templates for enabled Kafka events. Configure general templates or model-specific overrides.') . '</p>',
    ];

    // Get enabled events with full enhancement (models and templates)
    $enhanced_events = $this->templateManager->getFullyEnhancedEnabledTemplates();

    if (empty($enhanced_events)) {
      $form['empty_state'] = [
        '#type' => 'markup',
        '#markup' => '<div>' .
          '<p><strong>' . $this->t('No events enabled for Kafka publishing.') . '</strong></p>' .
          '<p>' . $this->t('Go to the <a href="@url">Enable Events</a> tab to add events first.', [
            '@url' => Url::fromRoute('eca_kafka.enable_events')->toString()
          ]) . '</p>' .
          '</div>',
      ];
      return $form;
    }

    // Build the templates table
    $form['templates_table'] = [
      '#type' => 'table',
      '#header' => [
        $this->t('Event'),
        $this->t('Models Using Event'),
        $this->t('Templates Status'),
        $this->t('Operations'),
      ],
      '#empty' => $this->t('No events are enabled for Kafka publishing.'),
      '#attributes' => ['class' => ['manage-templates-table']],
    ];

    foreach ($enhanced_events as $plugin_id => $event_data) {
      $form['templates_table'][$plugin_id] = $this->buildEventRow($plugin_id, $event_data);
    }

    return $form;
  }

  /**
   * Build a table row for an enabled event.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   * @param array $event_data
   *   The enhanced event data including templates and models.
   *
   * @return array
   *   The form row array.
   */
  protected function buildEventRow(string $plugin_id, array $event_data): array {
    // Use centralized formatting service for consistent display
    $display_data = $this->templateManager->formatEventDisplayData($plugin_id, $event_data);

    // Event column - consistent with EnableEventsForm
    $row['event'] = [
      '#type' => 'markup',
      '#markup' => sprintf(
        '<div class="event-info"><strong>%s</strong><br><small>%s | %s</small></div>',
        Html::escape($display_data['system_event_name']),
        Html::escape($display_data['plugin_id']),
        Html::escape($display_data['provider'])
      ),
    ];

    // Models column - use TemplateManager for standardized display
    $row['models'] = [
      '#type' => 'markup',
      '#markup' => $this->templateManager->getEventModelsSummary($event_data['models']),
    ];

    // Templates status column - use TemplateManager for standardized display
    $row['templates'] = [
      '#type' => 'markup',
      '#markup' => $this->templateManager->getTemplateStatusSummary($event_data['templates']),
    ];

    // Operations column - use TemplateManager for standardized operations
    // Pass the actual system_event_name value, not just the field reference
    $row['operations'] = $this->templateManager->buildEventRowOperations($event_data['system_event_name'], $event_data);

    return $row;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // This form is read-only, no submission needed
  }

}
