<?php

namespace Drupal\eca_kafka\Form;

use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\eca_kafka\Service\EcaEventInspector;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Url;
use Drupal\eca_kafka\Entity\KafkaMessageTemplate;
use Drupal\Component\Utility\Html;

/**
 * Form for resetting/deleting Kafka message templates.
 */
class TemplateResetForm extends ConfirmFormBase {

  /**
   * The ECA event inspector service.
   *
   * @var \Drupal\eca_kafka\Service\EcaEventInspector
   */
  protected EcaEventInspector $eventInspector;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The event plugin ID.
   *
   * @var string|null
   */
  protected ?string $eventPluginId = NULL;

  /**
   * The system event name.
   *
   * @var string|null
   */
  protected ?string $systemEventName = NULL;

  /**
   * Constructs a new TemplateResetForm.
   *
   * @param \Drupal\eca_kafka\Service\EcaEventInspector $event_inspector
   *   The ECA event inspector service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EcaEventInspector $event_inspector, EntityTypeManagerInterface $entity_type_manager) {
    $this->eventInspector = $event_inspector;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('eca_kafka.event_inspector'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'eca_kafka_template_reset_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?string $event_class_name = NULL): array {
    if (!$event_class_name) {
      $this->messenger()->addError($this->t('Event class name is required.'));
      return $form;
    }

    // The event_class_name parameter has been cleaned for URL safety
    // We need to convert it back to the actual system event name
    $clean_event_class = urldecode($event_class_name);
    $actual_system_event_name = $this->findActualSystemEventName($clean_event_class);
    
    if (!$actual_system_event_name) {
      $this->messenger()->addError($this->t('Event class not found: @name', ['@name' => $clean_event_class]));
      return [];
    }
    
    // Get plugin ID for display purposes
    $plugin_id = $this->eventInspector->getPluginIdFromSystemEventName($actual_system_event_name);
    if (!$plugin_id) {
      $this->messenger()->addError($this->t('Plugin ID not found for event class: @name', ['@name' => $actual_system_event_name]));
      return [];
    }
    
    $this->eventPluginId = $plugin_id;
    $this->systemEventName = $actual_system_event_name;

    // Get event information
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$this->eventPluginId] ?? [];
    $event_label = $event_definition['label'] ?? $this->eventPluginId;

    // Find existing templates
    $templates_info = $this->getTemplatesInfo($this->systemEventName);

    if (empty($templates_info['templates'])) {
      $this->messenger()->addWarning($this->t('No templates found for this event.'));
      return $this->redirect('eca_kafka.templates');
    }

    $form = parent::buildForm($form, $form_state);

    // Add event information
    $form['event_info'] = [
      '#type' => 'markup',
      '#markup' => '<div class="event-info"><strong>' . $this->t('Event: @label', ['@label' => $event_label]) . '</strong><br>' .
        '<small>' . $this->t('Plugin ID: @id', ['@id' => $this->eventPluginId]) . '</small></div>',
      '#weight' => -10,
    ];

    // Add templates summary
    $form['templates_summary'] = [
      '#type' => 'markup',
      '#markup' => $this->buildTemplatesSummary($templates_info),
      '#weight' => -5,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): string {
    return $this->t('Are you sure you want to clear all template content for this event?');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return $this->t('This action will clear the content of all templates (general and model-specific) for this event, but keep the event enabled for Kafka publishing. This action cannot be undone.');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): string {
    return $this->t('Clear Templates');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('eca_kafka.templates');
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    if (!$this->systemEventName) {
      return;
    }

    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    
    // Find all templates for this event
    $templates = $storage->loadByProperties([
      'system_event_name' => $this->systemEventName,
    ]);

    $reset_count = 0;
    $general_reset = FALSE;
    $model_specific_reset = 0;

    foreach ($templates as $template) {
      // Only reset templates that have content
      if (!empty($template->getMessageTemplate())) {
        // Clear the template content but keep the entity
        $template->set('message_template', '');
        $template->save();
        
        if ($template->getModelId()) {
          $model_specific_reset++;
        } else {
          $general_reset = TRUE;
        }
        $reset_count++;
      }
    }

    // Get event label for the message
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$this->eventPluginId] ?? [];
    $event_label = $event_definition['label'] ?? $this->eventPluginId;

    // Build success message
    if ($reset_count > 0) {
      $message_parts = [];
      if ($general_reset) {
        $message_parts[] = $this->t('general template');
      }
      if ($model_specific_reset > 0) {
        $message_parts[] = $this->formatPlural(
          $model_specific_reset,
          '1 model-specific template',
          '@count model-specific templates'
        );
      }

      $message = $this->t('Templates reset for "@event". Cleared: @items. The event remains enabled for Kafka publishing.', [
        '@event' => $event_label,
        '@items' => implode(' and ', $message_parts),
      ]);
    } else {
      $message = $this->t('No templates with content were found to reset for "@event".', [
        '@event' => $event_label,
      ]);
    }

    $this->messenger()->addStatus($message);

    // Redirect back to templates list
    $form_state->setRedirectUrl($this->getCancelUrl());
  }

  /**
   * Get information about existing templates for an event.
   *
   * @param string $system_event_name
   *   The system event name.
   *
   * @return array
   *   Array with template information.
   */
  protected function getTemplatesInfo(string $system_event_name): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadByProperties([
      'system_event_name' => $system_event_name,
    ]);

    $info = [
      'templates' => $templates,
      'general' => NULL,
      'model_specific' => [],
    ];

    foreach ($templates as $template) {
      if ($template->getModelId()) {
        $info['model_specific'][] = $template;
      } else {
        $info['general'] = $template;
      }
    }

    return $info;
  }

  /**
   * Build a summary of templates to be cleared.
   *
   * @param array $templates_info
   *   The templates information array.
   *
   * @return string
   *   HTML summary of templates.
   */
  protected function buildTemplatesSummary(array $templates_info): string {
    $items = [];

    // General template
    if ($templates_info['general'] && !empty($templates_info['general']->getMessageTemplate())) {
      $template = $templates_info['general'];
      $content_summary = $this->getContentSummary($template->getMessageTemplate());
      $items[] = '<li><strong>' . $this->t('General Template') . '</strong><br>' .
        '<small>' . $content_summary . '</small></li>';
    }

    // Model-specific templates
    foreach ($templates_info['model_specific'] as $template) {
      if (!empty($template->getMessageTemplate())) {
        $model_id = $template->getModelId();
        $model_labels = $this->eventInspector->getModelLabels([$model_id]);
        $model_label = $model_labels[$model_id] ?? $model_id;
        
        $content_summary = $this->getContentSummary($template->getMessageTemplate());
        $items[] = '<li><strong>' . $this->t('Model-specific: @model', ['@model' => $model_label]) . '</strong><br>' .
          '<small>' . $content_summary . '</small></li>';
      }
    }

    if (empty($items)) {
      return '<p>' . $this->t('No templates with content found.') . '</p>';
    }

    $count = count($items);
    $summary = '<p><strong>' . $this->formatPlural($count, 
      'The following template will be cleared:', 
      'The following @count templates will be cleared:'
    ) . '</strong></p>';

    return $summary . '<ul>' . implode('', $items) . '</ul>';
  }

  /**
   * Get a short summary of template content.
   *
   * @param string $content
   *   The template content.
   *
   * @return string
   *   A short summary.
   */
  protected function getContentSummary(string $content): string {
    $content = trim($content);
    
    if (empty($content)) {
      return $this->t('Empty template');
    }

    // Count lines and characters
    $lines = substr_count($content, "\n") + 1;
    $chars = strlen($content);

    // Show first line preview if not too long
    $first_line = strtok($content, "\n");
    if (strlen($first_line) > 50) {
      $first_line = substr($first_line, 0, 47) . '...';
    }

    return $this->t('@lines lines, @chars characters. Preview: "@preview"', [
      '@lines' => $lines,
      '@chars' => $chars,
      '@preview' => $first_line,
    ]);
  }

  /**
   * Find the actual system event name from a cleaned/URL-safe version.
   *
   * @param string $clean_event_class
   *   The cleaned event class name from the URL.
   *
   * @return string|null
   *   The actual system event name or NULL if not found.
   */
  protected function findActualSystemEventName(string $clean_event_class): ?string {
    // Get all templates and extract their system event names
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadMultiple();
    
    $system_event_names = [];
    foreach ($templates as $template) {
      $system_event_name = $template->getSystemEventName();
      if ($system_event_name && !in_array($system_event_name, $system_event_names)) {
        $system_event_names[] = $system_event_name;
      }
    }
    
    // Check exact match first
    if (in_array($clean_event_class, $system_event_names)) {
      return $clean_event_class;
    }
    
    // Check cleaned version match
    foreach ($system_event_names as $system_event_name) {
      if (Html::cleanCssIdentifier($system_event_name) === $clean_event_class) {
        return $system_event_name;
      }
    }
    
    return NULL;
  }

  /**
   * Find the actual plugin ID from a cleaned/URL-safe version.
   *
   * @param string $clean_plugin_id
   *   The cleaned plugin ID from the URL.
   *
   * @return string|null
   *   The actual plugin ID or NULL if not found.
   */
  protected function findActualPluginId(string $clean_plugin_id): ?string {
    // Get all available events
    $all_events = $this->eventInspector->getAllEvents();
    
    // First check if the clean ID matches exactly (no special chars)
    if (isset($all_events[$clean_plugin_id])) {
      return $clean_plugin_id;
    }
    
    // Search for a match by comparing cleaned versions
    foreach ($all_events as $plugin_id => $definition) {
      if (Html::cleanCssIdentifier($plugin_id) === $clean_plugin_id) {
        return $plugin_id;
      }
    }
    
    return NULL;
  }

}
