<?php

namespace Drupal\eca_kafka\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_kafka\Service\KafkaProducerService;
use Drupal\eca_kafka\Service\TemplateManagerService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Unified ECA action for sending messages to Kafka topics.
 */
#[Action(
  id: 'eca_kafka_send_message',
  label: new TranslatableMarkup('Kafka: Send message'),
  type: 'eca'
)]
#[EcaAction(
  description: new TranslatableMarkup('Sends a message to a specified Kafka topic. Can use templates from the template system or custom messages.'),
  version_introduced: '1.0.0',
)]
class KafkaSendMessage extends ConfigurableActionBase {

  /**
   * The Kafka producer service.
   *
   * @var \Drupal\eca_kafka\Service\KafkaProducerService
   */
  protected KafkaProducerService $kafkaProducer;

  /**
   * The template manager service.
   *
   * @var \Drupal\eca_kafka\Service\TemplateManagerService
   */
  protected TemplateManagerService $templateManager;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|null
   */
  protected ?ConfigFactoryInterface $configFactory = NULL;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->kafkaProducer = $container->get('eca_kafka.producer');
    $instance->templateManager = $container->get('eca_kafka.template_manager');
    $instance->setConfigFactory($container->get('config.factory'));
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
      'use_template' => TRUE,
      'topic_name' => '',
      'custom_message' => '',
      'message_key' => '',
      'headers' => '',
      'auto_create_topic' => FALSE,
      'result_token' => 'kafka_result',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['use_template'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use template'),
      '#description' => $this->t('If checked, will use the most specific template from the template system. If unchecked, will use the custom message below.'),
      '#default_value' => $this->configuration['use_template'],
    ];

    // Get available topics for dropdown
    $topic_options = $this->getTopicOptions();
    
    $form['topic_name'] = [
      '#type' => 'select',
      '#title' => $this->t('Kafka topic'),
      '#description' => $this->t('Select the Kafka topic to send the message to.'),
      '#options' => $topic_options,
      '#default_value' => $this->configuration['topic_name'],
      '#required' => TRUE,
      '#empty_option' => $this->t('- Select topic -'),
      '#eca_token_replacement' => TRUE,
      '#eca_token_reference' => TRUE,
    ];

    if (empty($topic_options)) {
      $form['topic_name']['#description'] = $this->t('No topics found. Make sure Kafka connection is properly configured. You can also type a topic name manually.');
      $form['topic_name']['#type'] = 'textfield';
      $form['topic_name']['#empty_option'] = NULL;
    }

    $form['custom_message'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Custom message'),
      '#description' => $this->t('Custom message content to send when "Use template" is not checked. Can be plain text, JSON, or any string format.'),
      '#default_value' => $this->configuration['custom_message'],
      '#eca_token_replacement' => TRUE,
      '#eca_token_reference' => TRUE,
      '#rows' => 8,
      '#states' => [
        'visible' => [
          ':input[name="use_template"]' => ['checked' => FALSE],
        ],
        'required' => [
          ':input[name="use_template"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['result_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Result token name'),
      '#description' => $this->t('Token name to store the operation result. Contains success status, response data, and error details.'),
      '#default_value' => $this->configuration['result_token'],
      '#required' => TRUE,
      '#eca_token_reference' => TRUE,
      '#element_validate' => [[$this, 'validateTokenName']],
    ];

    // Advanced options
    $form['advanced'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced options'),
      '#open' => FALSE,
    ];

    $form['advanced']['message_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Message key (optional)'),
      '#description' => $this->t('Optional key for message partitioning. Messages with the same key will go to the same partition.'),
      '#default_value' => $this->configuration['message_key'],
      '#eca_token_replacement' => TRUE,
      '#eca_token_reference' => TRUE,
    ];

    $form['advanced']['headers'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Headers (optional)'),
      '#description' => $this->t('Optional message headers in YAML format. Example:<br><code>X-Source: drupal<br>X-Event-Type: user_login</code>'),
      '#default_value' => $this->configuration['headers'],
      '#eca_token_replacement' => TRUE,
      '#eca_token_reference' => TRUE,
      '#rows' => 3,
    ];

    $form['advanced']['auto_create_topic'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Auto-create topic if it does not exist'),
      '#description' => $this->t('If checked, the topic will be automatically created if it does not exist. Uses default settings (1 partition, replication factor 3).'),
      '#default_value' => $this->configuration['auto_create_topic'],
    ];

    // Add help section about templates
    $form['template_help'] = [
      '#type' => 'details',
      '#title' => $this->t('Template information'),
      '#open' => FALSE,
      '#states' => [
        'visible' => [
          ':input[name="use_template"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['template_help']['info'] = [
      '#markup' => '<p>' . $this->t('When "Use template" is checked:') . '</p>' .
                  '<ul>' .
                  '<li>' . $this->t('A basic JSON template will be used with event name and timestamp') . '</li>' .
                  '<li>' . $this->t('For custom messages, uncheck "Use template" and use the custom message field below') . '</li>' .
                  '</ul>',
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::validateConfigurationForm($form, $form_state);

    // Validate that either template is used or custom message is provided
    $custom_message = (string) $form_state->getValue('custom_message');
    if (!$form_state->getValue('use_template') && empty(trim($custom_message))) {
      $form_state->setErrorByName('custom_message', $this->t('Custom message is required when not using templates.'));
    }

    // Validate headers format if provided
    $headers = trim((string) $form_state->getValue('headers'));
    if (!empty($headers)) {
      try {
        $this->parseHeaders($headers);
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('headers', $this->t('Invalid headers format: @error', ['@error' => $e->getMessage()]));
      }
    }
  }

  /**
   * Validates the token name field.
   *
   * @param array $element
   *   The form element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function validateTokenName(array $element, FormStateInterface $form_state): void {
    $value = trim($element['#value']);
    
    // Basic validation - just ensure it's not empty and doesn't contain spaces
    // ECA handles more complex token validation internally
    if (empty($value)) {
      $form_state->setError($element, $this->t('Token name cannot be empty.'));
      return;
    }
    
    // Check for spaces and invalid characters that could cause issues
    if (preg_match('/\s/', $value)) {
      $form_state->setError($element, $this->t('Token name cannot contain spaces.'));
      return;
    }
    
    // Check for potentially problematic characters
    if (preg_match('/[<>\'\"&]/', $value)) {
      $form_state->setError($element, $this->t('Token name contains invalid characters.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    try {
      // Get token service
      $token = $this->tokenService;

      // Replace tokens in configuration values
      $topic_name = trim($token->getOrReplace($this->configuration['topic_name']));
      $use_template = $this->configuration['use_template'];
      $custom_message = $token->getOrReplace($this->configuration['custom_message']);
      $message_key = trim($token->getOrReplace($this->configuration['message_key']));
      $headers_yaml = trim($token->getOrReplace($this->configuration['headers']));
      $auto_create_topic = $this->configuration['auto_create_topic'];
      $result_token = $this->configuration['result_token'];

      // Validate required values
      if (empty($topic_name)) {
        throw new \InvalidArgumentException('Topic name is required');
      }

      // Determine message content based on template setting
      if ($use_template) {
        // Template mode: use custom template (with token replacement) if available,
        // otherwise use basic default template
        $message_content = $this->getTemplateMessage();
      } else {
        // Custom message mode: use the message provided in the action configuration
        if (empty($custom_message)) {
          throw new \InvalidArgumentException('Custom message is required when not using templates');
        }
        $message_content = $custom_message;
      }

      // Parse headers if provided
      $headers = [];
      if (!empty($headers_yaml)) {
        $headers = $this->parseHeaders($headers_yaml);
      }

      // Build message data
      $message = [
        'value' => $message_content,
      ];

      if (!empty($message_key)) {
        $message['key'] = $message_key;
      }

      if (!empty($headers)) {
        $message['headers'] = $headers;
      }

      $this->logger->info('ECA Kafka: Attempting to send message to topic @topic (template: @template)', [
        '@topic' => $topic_name,
        '@template' => $use_template ? 'yes' : 'no',
      ]);

      // Check if topic exists and auto-create if needed
      if ($auto_create_topic) {
        $topic_info = $this->kafkaProducer->getTopicInfo($topic_name);
        if ($topic_info['success'] && !$topic_info['exists']) {
          $this->logger->info('ECA Kafka: Auto-creating topic @topic', [
            '@topic' => $topic_name,
          ]);

          $create_result = $this->kafkaProducer->createTopic($topic_name);
          if (!$create_result['success']) {
            throw new \RuntimeException("Failed to create topic '{$topic_name}': " . $create_result['error_message']);
          }

          $this->logger->info('ECA Kafka: Successfully created topic @topic', [
            '@topic' => $topic_name,
          ]);
        }
      }

      // Send the message
      $result = $this->kafkaProducer->sendMessage($topic_name, $message);

      // Structure results for ECA token system
      $token_data = [
        'success' => $result['success'],
        'topic_name' => $topic_name,
        'used_template' => $use_template,
        'message_size' => strlen($message_content),
        'status_code' => $result['status_code'],
        'offset' => $result['offset'],
        'partition' => $result['partition'],
        'timestamp' => $result['timestamp'],
        'error_code' => $result['error_code'],
        'error_message' => $result['error_message'],
        'headers_count' => count($headers),
        'has_key' => !empty($message_key),
        'processed_at' => date('c'),
      ];

      // Add result to token system for next actions
      $token->addTokenData($result_token, $token_data);

      if ($result['success']) {
        $this->logger->info('ECA Kafka: Successfully sent message to topic @topic. Offset: @offset, Partition: @partition', [
          '@topic' => $topic_name,
          '@offset' => $result['offset'] ?? 'unknown',
          '@partition' => $result['partition'] ?? 'unknown',
        ]);
      }
      else {
        $this->logger->error('ECA Kafka: Failed to send message to topic @topic: @error', [
          '@topic' => $topic_name,
          '@error' => $result['error_message'],
        ]);
      }

    }
    catch (\Exception $e) {
      // Error handling with structured token
      $error_data = [
        'success' => FALSE,
        'topic_name' => $topic_name ?? 'unknown',
        'used_template' => $use_template ?? FALSE,
        'message_size' => null,
        'status_code' => 0,
        'offset' => null,
        'partition' => null,
        'timestamp' => null,
        'error_code' => $e->getCode(),
        'error_message' => $e->getMessage(),
        'headers_count' => 0,
        'has_key' => FALSE,
        'processed_at' => date('c'),
      ];

      $token->addTokenData($result_token ?? 'kafka_result', $error_data);

      $this->logger->error('ECA Kafka: Action failed with error: @error', [
        '@error' => $e->getMessage(),
      ]);

      // Re-throw to indicate action failure
      throw $e;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    // Check if Kafka connection is configured
    $config = $this->getConfigFactory()->get('eca_kafka.connection');
    
    if (empty($config->get('kafka_rest_url')) || 
        empty($config->get('cluster_id')) || 
        empty($config->get('api_credentials'))) {
      $result = AccessResult::forbidden('Kafka connection is not fully configured');
    }
    else {
      $result = AccessResult::allowed();
    }

    return $return_as_object ? $result : $result->isAllowed();
  }

  /**
   * Gets available topic options for the form.
   *
   * @return array
   *   Array of topic options, keyed by topic name.
   */
  protected function getTopicOptions(): array {
    // Check if Kafka connection is configured first
    try {
      $config = $this->getConfigFactory()->get('eca_kafka.connection');
      
      if (empty($config->get('kafka_rest_url')) || 
          empty($config->get('cluster_id')) || 
          empty($config->get('api_credentials'))) {
        // Connection not configured - return empty array to trigger textfield fallback
        return [];
      }
      
      $topics_result = $this->kafkaProducer->listTopics();
      
      if ($topics_result['success'] && !empty($topics_result['topics'])) {
        $options = [];
        foreach ($topics_result['topics'] as $topic_name) {
          $options[$topic_name] = $topic_name;
        }
        return $options;
      }
      else {
        $this->logger->warning('ECA Kafka: Could not load topics list: @error', [
          '@error' => $topics_result['error_message'] ?? 'Unknown error',
        ]);
        return [];
      }
    }
    catch (\Exception $e) {
      $this->logger->error('ECA Kafka: Error loading topics for form: @error', [
        '@error' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Gets the message content from template system or generates basic template.
   *
   * @return string
   *   The message content to send.
   */
  protected function getTemplateMessage(): string {
    // Get ECA execution context
    $model_id = $this->ecaModelId ?? null;
    $event_class_name = $this->getEventClassName();
    
    if (!$event_class_name) {
      // Fallback if event class cannot be determined
      $event_class_name = 'UnknownEvent';
    }
    
    $this->logger->info('ECA Kafka: Looking for template with event_class_name="@event_class_name", model_id="@model_id"', [
      '@event_class_name' => $event_class_name,
      '@model_id' => $model_id ?? 'null',
    ]);
    
    // Get raw template content from TemplateManager using the class name
    $template_content = $this->templateManager->getTemplateContent($event_class_name, $model_id);
    
    $this->logger->info('ECA Kafka: Template content retrieved: "@content"', [
      '@content' => substr($template_content, 0, 200) . (strlen($template_content) > 200 ? '...' : ''),
    ]);
    
    // Let ECA handle token replacement through normal flow
    return $this->tokenService->replaceClear($template_content);
  }

  /**
   * Gets the event class name for template matching.
   *
   * @return string|null
   *   The event class name or NULL if not determinable.
   */
  protected function getEventClassName(): ?string {
    $original_event = $this->getEvent();
    if (!$original_event) {
      return NULL;
    }

    // Get the event class and extract just the class name (like in config forms)
    $event_class = get_class($original_event);
    $class_parts = explode('\\', $event_class);
    $class_name = end($class_parts);

    $this->logger->debug('ECA Kafka: Using event class name "@class_name" from full class "@full_class"', [
      '@class_name' => $class_name,
      '@full_class' => $event_class,
    ]);

    return $class_name;
  }

  /**
   * Parses headers from YAML-like format.
   *
   * @param string $headers_text
   *   Headers in YAML-like format.
   *
   * @return array
   *   Parsed headers array.
   *
   * @throws \Exception
   *   When headers format is invalid.
   */
  protected function parseHeaders(string $headers_text): array {
    $headers = [];
    $lines = explode("\n", $headers_text);
    
    foreach ($lines as $line_number => $line) {
      $line = trim($line);
      if (empty($line)) {
        continue;
      }

      // Simple key: value parsing
      if (!strpos($line, ':')) {
        throw new \Exception("Line " . ($line_number + 1) . " is missing colon separator: '{$line}'");
      }

      list($key, $value) = explode(':', $line, 2);
      $key = trim($key);
      $value = trim($value);

      if (empty($key)) {
        throw new \Exception("Line " . ($line_number + 1) . " has empty header name");
      }

      $headers[$key] = $value;
    }

    return $headers;
  }

  /**
   * Get the config factory.
   *
   * @return \Drupal\Core\Config\ConfigFactoryInterface
   *   The config factory.
   */
  public function getConfigFactory(): ConfigFactoryInterface {
    if (!isset($this->configFactory)) {
      // @phpstan-ignore-next-line
      $this->configFactory = \Drupal::configFactory();
    }
    return $this->configFactory;
  }

  /**
   * Set the config factory.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function setConfigFactory(ConfigFactoryInterface $config_factory): void {
    $this->configFactory = $config_factory;
  }

}
