<?php

namespace Drupal\eca_kafka\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\Component\Utility\Html;
use Drupal\eca_kafka\Entity\KafkaMessageTemplate;

/**
 * Service for managing Kafka message templates.
 * 
 * Centralizes all template CRUD operations to avoid code duplication
 * across forms and other components.
 */
class TemplateManagerService {

  use StringTranslationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The ECA event inspector service.
   *
   * @var \Drupal\eca_kafka\Service\EcaEventInspector
   */
  protected EcaEventInspector $eventInspector;

  /**
   * Constructs a new TemplateManagerService.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\eca_kafka\Service\EcaEventInspector $event_inspector
   *   The ECA event inspector service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, EcaEventInspector $event_inspector) {
    $this->entityTypeManager = $entity_type_manager;
    $this->eventInspector = $event_inspector;
  }

  /**
   * Enable an event by creating its general template.
   *
   * @param string $plugin_id
   *   The event plugin ID to enable.
   *
   * @return array
   *   Result array with 'success' (bool), 'message' (string), and optional 'template' (KafkaMessageTemplate).
   */
  public function enableEvent(string $plugin_id): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$plugin_id] ?? [];

    // Validate event definition exists and has event_class
    if (!isset($event_definition['event_class'])) {
      return [
        'success' => FALSE,
        'message' => $this->t('Event "@plugin_id" has no event_class definition and cannot be enabled.', [
          '@plugin_id' => $plugin_id,
        ]),
      ];
    }

    // Extract class name as system_event_name
    $system_event_name = $this->getSystemEventNameFromPluginId($plugin_id);
    $template_id = KafkaMessageTemplate::generateId($system_event_name);

    // Check if template already exists
    if ($storage->load($template_id)) {
      return [
        'success' => FALSE,
        'message' => $this->t('Event "@plugin_id" (@class_name) is already enabled.', [
          '@plugin_id' => $plugin_id,
          '@class_name' => $system_event_name,
        ]),
      ];
    }

    try {
      // Create new general template
      $template = $storage->create([
        'id' => $template_id,
        'label' => ($event_definition['label'] ?? $plugin_id) . ' Template',
        'system_event_name' => $system_event_name,
        'model_id' => NULL, // General template
        'message_template' => '', // Empty template to be configured later
      ]);

      $template->save();

      return [
        'success' => TRUE,
        'message' => $this->t('Successfully enabled event "@plugin_id" (@class_name) for Kafka publishing.', [
          '@plugin_id' => $plugin_id,
          '@class_name' => $system_event_name,
        ]),
        'template' => $template,
      ];

    } catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => $this->t('Failed to create template for event "@plugin_id": @error', [
          '@plugin_id' => $plugin_id,
          '@error' => $e->getMessage(),
        ]),
      ];
    }
  }

  /**
   * Disable an event by deleting all its templates.
   *
   * @param string $plugin_id
   *   The event plugin ID to disable.
   *
   * @return array
   *   Result array with 'success' (bool), 'message' (string), and 'deleted_count' (int).
   */
  public function disableEvent(string $plugin_id): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$plugin_id] ?? [];

    // Validate event definition exists
    if (!isset($event_definition['event_class'])) {
      return [
        'success' => FALSE,
        'message' => $this->t('Could not find event class for plugin ID @plugin_id', [
          '@plugin_id' => $plugin_id,
        ]),
        'deleted_count' => 0,
      ];
    }

    // Extract class name
    $system_event_name = $this->getSystemEventNameFromPluginId($plugin_id);

    // Find all templates for this event (both general and model-specific)
    $templates = $storage->loadByProperties([
      'system_event_name' => $system_event_name,
    ]);

    if (empty($templates)) {
      return [
        'success' => FALSE,
        'message' => $this->t('No templates found for the specified event.'),
        'deleted_count' => 0,
      ];
    }

    $deleted_count = 0;
    $event_label = $event_definition['label'] ?? $plugin_id;

    // Delete all templates for this event
    foreach ($templates as $template) {
      $template->delete();
      $deleted_count++;
    }

    return [
      'success' => TRUE,
      'message' => $this->t('Successfully disabled event "@event_label". Removed @count template(s).', [
        '@event_label' => $event_label,
        '@count' => $deleted_count,
      ]),
      'deleted_count' => $deleted_count,
    ];
  }

  /**
   * Get all enabled templates keyed by plugin ID.
   *
   * @return array
   *   Array of enabled template data keyed by plugin ID.
   *   Each value contains:
   *   - plugin_id: The ECA plugin ID
   *   - template_id: The template entity ID
   *   - label: The template label
   *   - system_event_name: The class name used internally
   *   - template: The template entity
   */
  public function getEnabledTemplates(): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadMultiple();

    $enabled = [];
    foreach ($templates as $template) {
      $system_event_name = $template->getSystemEventName();
      // Only consider general templates (no model_id) for enabled events display
      // Model-specific templates will be handled separately
      if ($system_event_name && !$template->getModelId()) {
        // Convert system event name (class name) back to plugin ID
        $plugin_id = $this->eventInspector->getPluginIdFromSystemEventName($system_event_name);
        
        if ($plugin_id) {
          $enabled[$plugin_id] = [
            'plugin_id' => $plugin_id,
            'template_id' => $template->id(),
            'label' => $template->label(),
            'system_event_name' => $system_event_name,
            'template' => $template,
          ];
        }
      }
    }

    return $enabled;
  }

  /**
   * Get template for event/model combination.
   *
   * @param string $system_event_name
   *   The system event name (class name).
   * @param string|null $model_id
   *   The model ID (optional).
   *
   * @return \Drupal\eca_kafka\Entity\KafkaMessageTemplate|null
   *   The best matching template or NULL if none found.
   */
  public function getTemplate(string $system_event_name, ?string $model_id = NULL): ?KafkaMessageTemplate {
    return $this->eventInspector->findBestTemplate($system_event_name, $model_id);
  }

  /**
   * Get template message content for given event and model.
   *
   * @param string $system_event_name
   *   The system event name (class name).
   * @param string|null $model_id
   *   The model ID (optional).
   *
   * @return string
   *   The raw template content (with tokens) or basic JSON template.
   */
  public function getTemplateContent(string $system_event_name, ?string $model_id = NULL): string {
    // Get best matching template (model-specific → general → null)
    $template = $this->getTemplate($system_event_name, $model_id);
    
    if ($template && !empty($template->getMessageTemplate())) {
      // Return raw template content with tokens intact
      return $template->getMessageTemplate();
    }
    
    // No template found - return basic Kafka-standard JSON template
    return $this->generateBasicKafkaTemplate($system_event_name);
  }

  /**
   * Generate basic Kafka-standard JSON template with tokens.
   *
   * @param string $system_event_name
   *   The system event name.
   *
   * @return string
   *   Basic JSON template following Kafka standards with ECA tokens.
   */
  protected function generateBasicKafkaTemplate(string $system_event_name): string {
    $template_data = [
      'event' => $system_event_name,
      'timestamp' => '[current-date:custom:c]',  // ISO 8601 token
      'source' => 'drupal-eca',
      'data' => null,
    ];

    return json_encode($template_data, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
  }

  /**
   * Check if a template exists for the given event/model combination.
   *
   * @param string $system_event_name
   *   The system event name (class name).
   * @param string|null $model_id
   *   The model ID (optional).
   *
   * @return bool
   *   TRUE if a template exists, FALSE otherwise.
   */
  public function hasTemplate(string $system_event_name, ?string $model_id = NULL): bool {
    return $this->eventInspector->hasTemplate($system_event_name, $model_id);
  }

  /**
   * Create or update a template.
   *
   * @param array $template_data
   *   Template data array with keys:
   *   - id: Template ID (optional for new templates)
   *   - label: Human-readable label
   *   - system_event_name: Class name
   *   - model_id: Model ID (optional)
   *   - message_template: Message template content
   *
   * @return array
   *   Result array with 'success' (bool), 'message' (string), and optional 'template' (KafkaMessageTemplate).
   */
  public function saveTemplate(array $template_data): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');

    try {
      // Generate ID if not provided
      if (empty($template_data['id'])) {
        $template_data['id'] = KafkaMessageTemplate::generateId(
          $template_data['system_event_name'],
          $template_data['model_id'] ?? NULL
        );
      }

      // Load existing template or create new one
      $template = $storage->load($template_data['id']);
      if ($template) {
        // Update existing template using proper method names
        if (isset($template_data['label'])) {
          $template->set('label', $template_data['label']);
        }
        if (isset($template_data['system_event_name'])) {
          $template->setSystemEventName($template_data['system_event_name']);
        }
        if (array_key_exists('model_id', $template_data)) {
          $template->setModelId($template_data['model_id']);
        }
        if (isset($template_data['message_template'])) {
          $template->setMessageTemplate($template_data['message_template']);
        }
        $action = 'updated';
      } else {
        // Create new template
        $template = $storage->create($template_data);
        $action = 'created';
      }

      $template->save();

      return [
        'success' => TRUE,
        'message' => $this->t('Template successfully @action.', ['@action' => $action]),
        'template' => $template,
      ];

    } catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => $this->t('Failed to save template: @error', [
          '@error' => $e->getMessage(),
        ]),
      ];
    }
  }

  /**
   * Delete a template by its ID.
   *
   * @param string $template_id
   *   The template ID to delete.
   *
   * @return array
   *   Result array with 'success' (bool) and 'message' (string).
   */
  public function deleteTemplate(string $template_id): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $template = $storage->load($template_id);

    if (!$template) {
      return [
        'success' => FALSE,
        'message' => $this->t('Template not found.'),
      ];
    }

    try {
      $label = $template->label();
      $template->delete();

      return [
        'success' => TRUE,
        'message' => $this->t('Template "@label" successfully deleted.', [
          '@label' => $label,
        ]),
      ];

    } catch (\Exception $e) {
      return [
        'success' => FALSE,
        'message' => $this->t('Failed to delete template: @error', [
          '@error' => $e->getMessage(),
        ]),
      ];
    }
  }

  /**
   * Get system event name (class name) from plugin ID.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   *
   * @return string
   *   The system event name (class name).
   */
  public function getSystemEventNameFromPluginId(string $plugin_id): string {
    return $this->eventInspector->getSystemEventNameFromPluginId($plugin_id) ?? $plugin_id;
  }

  /**
   * Get plugin ID from system event name (class name).
   *
   * @param string $system_event_name
   *   The system event name (class name).
   *
   * @return string|null
   *   The plugin ID or NULL if not found.
   */
  public function getPluginIdFromSystemEventName(string $system_event_name): ?string {
    return $this->eventInspector->getPluginIdFromSystemEventName($system_event_name);
  }

  /**
   * Get enabled templates with enhanced event metadata.
   *
   * @return array
   *   Array of enabled template data keyed by plugin ID with full metadata:
   *   - plugin_id: The ECA plugin ID
   *   - template_id: The template entity ID
   *   - label: The enhanced event label
   *   - system_event_name: The class name used internally
   *   - provider: The event provider module
   *   - template: The template entity
   */
  public function getEnhancedEnabledTemplates(): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadMultiple();
    $all_events = $this->eventInspector->getAllEvents();

    $enabled = [];
    foreach ($templates as $template) {
      $system_event_name = $template->getSystemEventName();
      // Only consider general templates (no model_id) for enabled events display
      if ($system_event_name && !$template->getModelId()) {
        // Convert system event name (class name) back to plugin ID
        $plugin_id = $this->eventInspector->getPluginIdFromSystemEventName($system_event_name);
        
        if ($plugin_id) {
          $event_definition = $all_events[$plugin_id] ?? [];
          
          $enabled[$plugin_id] = [
            'plugin_id' => $plugin_id,
            'template_id' => $template->id(),
            'label' => $template->label() ?: $event_definition['label'] ?: $system_event_name,
            'system_event_name' => $system_event_name,
            'provider' => $event_definition['provider'] ?? 'Unknown',
            'template' => $template,
          ];
        }
      }
    }

    return $enabled;
  }

  /**
   * Get all available events formatted for JavaScript use.
   *
   * @return array
   *   Array of event data for JavaScript, excluding already enabled events.
   */
  public function getAvailableEventsForJs(): array {
    $event_usage_mapping = $this->eventInspector->getEventUsageMapping();
    $enabled_events = $this->getEnabledTemplates();

    $events_for_js = [];
    foreach ($event_usage_mapping as $plugin_id => $usage_data) {
      // Skip if already enabled for Kafka
      if (isset($enabled_events[$plugin_id])) {
        continue;
      }

      $definition = $usage_data['definition'];
      $events_for_js[] = [
        'plugin_id' => $plugin_id,
        'label' => $definition['label'] ?? $plugin_id,
        'event_name' => $usage_data['event_name'] ?? 'N/A',
        'provider' => $definition['provider'] ?? 'Unknown',
        'is_used' => $usage_data['is_used'],
        'search_text' => strtolower($plugin_id . ' ' . ($definition['label'] ?? '')),
      ];
    }

    return $events_for_js;
  }

  /**
   * Enable multiple events and return aggregated results.
   *
   * @param array $plugin_ids
   *   Array of plugin IDs to enable.
   *
   * @return array
   *   Aggregated result with counts and categorized messages:
   *   - success_count: Number of successfully enabled events
   *   - skipped_messages: Array of skip messages
   *   - error_messages: Array of error messages
   *   - success: Overall success boolean
   */
  public function enableMultipleEvents(array $plugin_ids): array {
    $result = [
      'success_count' => 0,
      'skipped_messages' => [],
      'error_messages' => [],
      'success' => FALSE,
    ];

    foreach ($plugin_ids as $plugin_id) {
      $plugin_id = trim($plugin_id);
      if (empty($plugin_id)) {
        continue;
      }

      $enable_result = $this->enableEvent($plugin_id);
      
      if ($enable_result['success']) {
        $result['success_count']++;
      } elseif (strpos($enable_result['message'], 'already enabled') !== FALSE) {
        $result['skipped_messages'][] = $enable_result['message'];
      } else {
        $result['error_messages'][] = $enable_result['message'];
      }
    }

    // Overall success if at least one event was enabled and no errors
    $result['success'] = $result['success_count'] > 0 && empty($result['error_messages']);

    return $result;
  }

  /**
   * Format event data for table display.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   * @param array $event_data
   *   The enhanced event data from getEnhancedEnabledTemplates().
   *
   * @return array
   *   Formatted data for display:
   *   - display_label: Primary label for display
   *   - system_event_name: System event name
   *   - provider: Provider module name
   *   - plugin_id: Original plugin ID
   */
  public function formatEventDisplayData(string $plugin_id, array $event_data): array {
    return [
      'display_label' => $event_data['label'] ?? $event_data['system_event_name'] ?? $plugin_id,
      'system_event_name' => $event_data['system_event_name'] ?? $plugin_id,
      'provider' => $event_data['provider'] ?? 'Unknown',
      'plugin_id' => $plugin_id,
    ];
  }

  /**
   * Get template statistics.
   *
   * @return array
   *   Array with template counts and statistics.
   */
  public function getTemplateStats(): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadMultiple();

    $stats = [
      'total' => count($templates),
      'general' => 0,
      'model_specific' => 0,
      'configured' => 0,
      'empty' => 0,
    ];

    foreach ($templates as $template) {
      if ($template->getModelId()) {
        $stats['model_specific']++;
      } else {
        $stats['general']++;
      }

      if (trim($template->getMessageTemplate())) {
        $stats['configured']++;
      } else {
        $stats['empty']++;
      }
    }

    return $stats;
  }

  /**
   * Get template status summary as HTML markup.
   *
   * @param array $templates
   *   Array of templates for an event with structure:
   *   - general: General template data or null
   *   - model_specific: Array of model-specific templates keyed by model_id
   *
   * @return string
   *   HTML markup for templates status display.
   */
  public function getTemplateStatusSummary(array $templates): string {
    $general_template = $templates['general'] ?? null;
    $model_specific = $templates['model_specific'] ?? [];

    $status_parts = [];

    // General template status
    if ($general_template && !empty($general_template['message_template'])) {
      $status_parts[] = '<strong>General:</strong> Configured';
    } else {
      $status_parts[] = '<strong>General:</strong> Empty';
    }

    // Model-specific templates status - Show individual model details
    if (!empty($model_specific)) {
      $configured_models = [];
      $empty_models = [];
      
      // Get model labels for display
      $model_ids = array_keys($model_specific);
      $model_labels = $this->eventInspector->getModelLabels($model_ids);
      
      // Categorize models by template status
      foreach ($model_specific as $model_id => $template) {
        $model_label = $model_labels[$model_id] ?? $model_id;
        if (!empty($template['message_template'])) {
          $configured_models[] = $model_label;
        } else {
          $empty_models[] = $model_label;
        }
      }
      
      // Build detailed status with model names
      $model_status_parts = [];
      
      if (!empty($configured_models)) {
        if (count($configured_models) === 1) {
          $model_status_parts[] = '<strong>' . 
            htmlspecialchars($configured_models[0]) . ':</strong> configured';
        } else {
          $model_status_parts[] = '<strong>' . 
            htmlspecialchars(implode(', ', $configured_models)) . ':</strong> configured';
        }
      }
      
      if (!empty($empty_models)) {
        if (count($empty_models) === 1) {
          $model_status_parts[] = '<strong>' . 
            htmlspecialchars($empty_models[0]) . ':</strong> empty';
        } else {
          $model_status_parts[] = '<strong>' . 
            htmlspecialchars(implode(', ', $empty_models)) . ':</strong> empty';
        }
      }
      
      if (!empty($model_status_parts)) {
        $status_parts[] = '<div><strong>Model-specific:</strong><br>' . 
          implode('<br>', $model_status_parts) . '</div>';
      }
    }

    return '<div>' . implode('<br>', $status_parts) . '</div>';
  }

  /**
   * Get event models usage summary as HTML markup.
   *
   * @param array $models
   *   Array of models using an event, keyed by model_id with model info.
   *
   * @return string
   *   HTML markup for models column display.
   */
  public function getEventModelsSummary(array $models): string {
    if (empty($models)) {
      return '<div><em>Not used in any models</em></div>';
    }

    $model_list = [];
    foreach ($models as $model_id => $model_info) {
      $model_list[] = sprintf(
        '<div>%s</div>',
        htmlspecialchars($model_info['label'] ?? $model_id)
      );
    }

    $count = count($models);
    $summary = $count === 1 ? 
      $this->t('Used in 1 model:') : 
      $this->t('Used in @count models:', ['@count' => $count]);

    return sprintf(
      '<div><strong>%s</strong><br>%s</div>',
      $summary,
      implode('', $model_list)
    );
  }

  /**
   * Build event row operations for template management tables.
   *
   * @param string $system_event_name
   *   The system event name.
   * @param array $event_data
   *   The event data including templates and models.
   *
   * @return array
   *   Form API operations element array.
   */
  public function buildEventRowOperations(string $system_event_name, array $event_data): array {
    $operations = [];
    
    // Clean the system event name for URLs using Drupal's built-in function
    // This converts special characters like dots and colons to underscores
    $clean_system_event_name = Html::cleanCssIdentifier($system_event_name);
    
    // Edit operation - always available
    $operations['edit'] = [
      'title' => $this->t('Edit Templates'),
      'url' => Url::fromRoute('eca_kafka.template_edit', [
        'event_class_name' => $clean_system_event_name,
      ]),
    ];
    
    // Add reset operation if ANY templates exist (general OR model-specific)
    if ($this->hasEventTemplateContent($event_data['templates'])) {
      $operations['reset'] = [
        'title' => $this->t('Reset Templates'),
        'url' => Url::fromRoute('eca_kafka.template_reset', [
          'event_class_name' => $clean_system_event_name,
        ]),
        'attributes' => [
          'class' => ['use-ajax'],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => json_encode(['width' => 500]),
        ],
      ];
    }
    
    return [
      '#type' => 'operations',
      '#links' => $operations,
      '#prefix' => '<div class="kafka-dropbutton">',
      '#suffix' => '</div>',
    ];
  }

  /**
   * Get event display markup for table rows.
   *
   * @param string $system_event_name
   *   The system event name.
   *
   * @return string
   *   HTML markup for event display column.
   */
  public function getEventDisplayMarkup(string $system_event_name): string {
    $plugin_id = $this->getPluginIdFromSystemEventName($system_event_name);
    $all_events = $this->eventInspector->getAllEvents();
    $event_definition = $all_events[$plugin_id] ?? [];
    
    $label = $event_definition['label'] ?? $plugin_id ?? $system_event_name;
    $provider = $event_definition['provider'] ?? 'Unknown';

    return sprintf(
      '<div><strong>%s</strong><br><small>%s | %s</small></div>',
      htmlspecialchars($label),
      htmlspecialchars($plugin_id ?? $system_event_name),
      htmlspecialchars($provider)
    );
  }

  /**
   * Get enabled events with their template and model information.
   *
   * @return array
   *   Array of enabled events with template and model data keyed by system event name.
   *   Each entry contains:
   *   - system_event_name: The system event name
   *   - templates: Array with 'general' and 'model_specific' template data
   *   - models: Array of models using this event
   */
  public function getEnabledEventsWithTemplatesAndModels(): array {
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    $templates = $storage->loadMultiple();

    $enabled_events = [];

    // First, collect ALL templates and organize by event
    foreach ($templates as $template) {
      $system_event_name = $template->getSystemEventName();
      if (!$system_event_name) {
        continue;
      }
      
      // Initialize event if not seen before
      if (!isset($enabled_events[$system_event_name])) {
        $enabled_events[$system_event_name] = [
          'system_event_name' => $system_event_name,
          'templates' => [
            'general' => null,
            'model_specific' => [],
          ],
          'models' => [],
        ];
      }
      
      // Categorize template
      if ($template->getModelId()) {
        // Model-specific template
        $enabled_events[$system_event_name]['templates']['model_specific'][$template->getModelId()] = [
          'id' => $template->id(),
          'label' => $template->label(),
          'model_id' => $template->getModelId(),
          'message_template' => $template->getMessageTemplate(),
        ];
      } else {
        // General template
        $enabled_events[$system_event_name]['templates']['general'] = [
          'id' => $template->id(),
          'label' => $template->label(),
          'message_template' => $template->getMessageTemplate(),
        ];
      }
    }

    // Add model information for each enabled event
    $event_models_mapping = $this->eventInspector->getEventModelsMapping();
    foreach ($enabled_events as $system_event_name => &$event_data) {
      // Direct lookup using system event names as keys
      if (isset($event_models_mapping[$system_event_name])) {
        $event_data['models'] = $event_models_mapping[$system_event_name];
      }
    }

    return $enabled_events;
  }

  /**
   * Get fully enhanced enabled templates with model information and template details.
   *
   * This method provides complete information needed for the ManageTemplatesForm,
   * including model usage and template status details.
   *
   * @return array
   *   Array of enhanced template data keyed by plugin ID.
   *   Each entry contains:
   *   - plugin_id: The ECA plugin ID
   *   - template_id: The general template entity ID
   *   - label: The template/event label
   *   - system_event_name: The class name used internally
   *   - provider: The provider module name
   *   - template: The general template entity
   *   - models: Array of models using this event (from unified mapping)
   *   - templates: Array with 'general' and 'model_specific' template details
   */
  public function getFullyEnhancedEnabledTemplates(): array {
    // Get unified event usage mapping
    $event_usage_mapping = $this->eventInspector->getEventUsageMapping();
    $enabled_events = $this->getEnhancedEnabledTemplates();
    $storage = $this->entityTypeManager->getStorage('kafka_message_template');
    
    $enhanced_events = [];
    
    // Enhance each enabled event with full template and model information
    foreach ($enabled_events as $plugin_id => $event_data) {
      $system_event_name = $event_data['system_event_name'];
      $usage_data = $event_usage_mapping[$plugin_id] ?? [];
      
      // Initialize enhanced event data with unified information
      $enhanced_events[$plugin_id] = $event_data;
      $enhanced_events[$plugin_id]['templates'] = [
        'general' => null,
        'model_specific' => [],
      ];
      // Use models from unified mapping (handles key translation automatically)
      $enhanced_events[$plugin_id]['models'] = $usage_data['models'] ?? [];
      
      // Get all templates for this event (general + model-specific)
      $templates = $storage->loadByProperties([
        'system_event_name' => $system_event_name,
      ]);
      
      // Categorize templates by type
      foreach ($templates as $template) {
        if ($template->getModelId()) {
          // Model-specific template
          $enhanced_events[$plugin_id]['templates']['model_specific'][$template->getModelId()] = [
            'id' => $template->id(),
            'label' => $template->label(),
            'model_id' => $template->getModelId(),
            'message_template' => $template->getMessageTemplate(),
          ];
        } else {
          // General template (should be the same as $event_data['template'])
          $enhanced_events[$plugin_id]['templates']['general'] = [
            'id' => $template->id(),
            'label' => $template->label(),
            'message_template' => $template->getMessageTemplate(),
          ];
        }
      }
    }
    
    return $enhanced_events;
  }

  /**
   * Check if event has any template content (general or model-specific).
   *
   * @param array $templates
   *   Array of templates with 'general' and 'model_specific' keys.
   *
   * @return bool
   *   TRUE if any template has content, FALSE otherwise.
   */
  protected function hasEventTemplateContent(array $templates): bool {
    // Check for general template content
    if (!empty($templates['general']['message_template'])) {
      return TRUE;
    }
    
    // Check for model-specific templates with content
    if (!empty($templates['model_specific'])) {
      foreach ($templates['model_specific'] as $template) {
        if (!empty($template['message_template'])) {
          return TRUE;
        }
      }
    }
    
    return FALSE;
  }

}
