<?php

namespace Drupal\eca_ng\Controller;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\MessageCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\eca\Entity\Eca;
use Drupal\modeler_api\Form\Settings;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for templates in ECA.
 */
class Template extends ControllerBase {

  public function __construct(
    private Request $request,
    private TimeInterface $time,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): Template {
    return new Template(
      $container->get('request_stack')->getCurrentRequest(),
      $container->get('datetime.time'),
    );
  }

  /**
   * Apply a template to an element.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The response that contains a message.
   */
  public function apply(): AjaxResponse {
    $frontEndData = $this->request->getContent();
    [$templateId, $eventId, $ecaId, $select, $data] = json_decode($frontEndData, TRUE);
    $eca = Eca::load($templateId);
    $elements = [];
    $ts = $this->time->getRequestTime();
    foreach ($eca->getAllEventElements($eventId) as $id => $type) {
      $element = $eca->get($type . 's')[$id];
      $config = $element['configuration'] ?? NULL;
      if ($type === 'event') {
        $element['applied_templates'][] = $frontEndData;
      }
      else {
        $id .= '_' . $ts;
      }
      if (is_array($config)) {
        if (isset($data[$id])) {
          $config = array_merge($config, $data[$id]);
        }
        else {
          foreach ($config as $configKey => $configValue) {
            if (!is_scalar($configValue)) {
              continue;
            }
            if (($pos = strpos($configValue, '[eca-template:select:')) !== FALSE) {
              if (($offset = strpos($configValue, ']', $pos)) !== FALSE) {
                $token = substr($configValue, $pos, $offset - $pos + 1);
                $configValue = str_replace($token, $select, $configValue);
              }
            }
            foreach ($data as $key => $value) {
              $configValue = str_replace('[' . $key . ']', is_array($value) ? implode(',', $value) : $value, $configValue);
            }
            $config[$configKey] = $configValue;
          }
        }
        $element['configuration'] = $config;
      }
      foreach ($element['successors'] ?? [] as $key => $successor) {
        $element['successors'][$key]['id'] .= '_' . $ts;
        if ($element['successors'][$key]['condition'] !== '') {
          $element['successors'][$key]['condition'] .= '_' . $ts;
        }
      }
      $elements[$type . 's'][$id] = $element;
    }
    $newEca = Eca::load($ecaId);
    if (!$newEca) {
      $newEca = Eca::create([
        'id' => $ecaId,
        'weight' => 0,
        'template' => FALSE,
      ]);
      $newEca->setThirdPartySetting('modeler_api', 'label', ucfirst(str_replace('_', ' ', $ecaId)));
      $newEca->setThirdPartySetting('modeler_api', 'storage', Settings::STORAGE_OPTION_NONE);
    }
    foreach ($elements as $type => $elementsByType) {
      $elementsByType = NestedArray::mergeDeep($newEca->get($type) ?? [], $elementsByType);
      $newEca->set($type, $elementsByType);
    }
    $newEca->save();

    $response = new AjaxResponse();
    $response->addCommand(new MessageCommand('Automation template saved and applied.'));
    return $response;
  }

}
