<?php

namespace Drupal\eca_ng\Hook;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Hook\Order\Order;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\State\StateInterface;
use Drupal\eca\Entity\Eca;
use Drupal\eca\Event\TriggerEvent;
use Drupal\eca\PluginManager\Event;
use Drupal\eca_ng\Service\Template;
use Drupal\eca_form\Event\FormBuild;
use Drupal\eca_form\Event\FormValidate;

/**
 * Implements form hooks for templates.
 */
class TemplateFormHooks {

  /**
   * Constructs a new FormHooks object.
   */
  public function __construct(
    protected TriggerEvent $triggerEvent,
    protected StateInterface $state,
    protected Event $eventPluginManager,
    protected AccountProxyInterface $currentUser,
    protected Template $templateService,
  ) {}

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter', order: Order::Last)]
  public function formAlter(array &$form, FormStateInterface $form_state): void {
    if (!isset($form['#form_id']) || !$this->currentUser->hasPermission('modeler api edit eca')) {
      return;
    }
    $settings = [];
    $templates = $this->state->get('eca.templates', []);
    $events = [
      'eca.form.build' => new FormBuild($form, $form_state),
      'eca.form.validate' => new FormValidate($form, $form_state),
    ];
    foreach ($events as $eventName => $event) {
      $eventPluginId = $this->eventPluginManager->getPluginIdForSystemEvent($eventName);
      $eventPluginClass = $this->eventPluginManager->getDefinition($eventPluginId)['class'];
      $event = new FormBuild($form, $form_state);
      foreach ($templates[$eventName] ?? [] as $ecaId => $events) {
        $eca = Eca::load($ecaId);
        foreach ($events as $eventId => $wildcard) {
          if (call_user_func($eventPluginClass . '::appliesForWildcard', $event, $eventName, $wildcard)) {
            $select = NULL;
            $config = [];
            if ($wildcard === '*:*:*:*') {
              $config['hidden'][$eventId]['form_ids'] = $form['#form_id'];
              $newId = 'form_' . $form['#form_id'];
            }
            else {
              $newId = 'form_' . str_replace(['*', ':'], ['any', '_'], $wildcard);
            }
            foreach ($eca->getAllEventElements($eventId) as $id => $type) {
              foreach ($eca->get($type . 's')[$id]['configuration'] ?? [] as $value) {
                if (!is_scalar($value) || !str_contains($value, '[eca-template:')) {
                  continue;
                }
                $offset = 0;
                while (($pos = strpos($value, '[eca-template:', $offset)) !== FALSE) {
                  $offset = strpos($value, ']', $pos);
                  if ($offset === FALSE) {
                    break;
                  }
                  $token = substr($value, $pos + 1, $offset - $pos - 1);
                  $tokenParts = explode(':', $token);
                  $phase = $tokenParts[1];
                  $context = $tokenParts[2];
                  unset($tokenParts[0], $tokenParts[1]);
                  $subToken = implode(':', $tokenParts);
                  if ($context === 'form' && $phase === 'select') {
                    if ($select === NULL) {
                      $select = $subToken;
                    }
                    else {
                      // This is an error as a template can only have 1 select
                      // token.
                      continue;
                    }
                  }
                  elseif ($phase === 'config') {
                    if ($context === 'form' || $context === 'global') {
                      $nested = [];
                      NestedArray::setValue($nested, explode(':', $subToken), $token);
                      $config = NestedArray::mergeDeep($config, $nested);
                    }
                  }
                }
              }
            }
            if ($select !== NULL) {
              $settings[] = [
                'label' => $eca->get('events')[$eventId]['label'],
                'select' => $select,
                'config' => $config,
                'template_id' => $ecaId,
                'event_id' => $eventId,
                'eca_id' => $newId,
              ];
            }
          }
        }
      }
    }
    if ($settings) {
      $form['ecatemplatedummy'] = [
        '#type' => 'textfield',
        '#attributes' => [
          'class' => ['hidden'],
        ],
        '#ajax' => [
          'callback' => [self::class, 'ajaxCallback'],
          'wrapper' => $form['#id'],
          'event' => 'eca-ajax-reload',
        ],
      ];
      $form['#attached']['library'][] = 'eca_ng/templates.forms';
      $form['#attached']['drupalSettings']['eca_form']['templates'] = $settings;
      $this->templateService->enableEditing();
    }
  }

  /**
   * Dummy method for the pseudo form reload.
   */
  public static function ajaxCallback(array &$form, FormStateInterface $form_state): array {
    return $form;
  }

}
