<?php

declare(strict_types=1);

namespace Drupal\eca_ng\Plugin\TopBarItem;

use Drupal\Core\Htmx\Htmx;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Url;
use Drupal\eca\Processor;
use Drupal\eca_ng\Service\Template;
use Drupal\navigation\Attribute\TopBarItem;
use Drupal\navigation\TopBarItemBase;
use Drupal\navigation\TopBarRegion;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the top bar item to turn template editing on and off.
 *
 * @internal
 */
#[TopBarItem(
  id: 'eca_template_editing',
  region: TopBarRegion::Actions,
  label: new TranslatableMarkup('Edit automations'),
)]
final class TemplateEditing extends TopBarItemBase implements ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * The template service.
   *
   * @var \Drupal\eca_ng\Service\Template
   */
  protected Template $templateService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): TemplateEditing {
    $instance = new static($configuration, $plugin_id, $plugin_definition);
    $instance->templateService = $container->get('eca_ng.service.template');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $build = [
      '#cache' => [
        'contexts' => ['route'],
      ],
    ];
    $processDebuggers = Processor::getAppliedEvents();
    if (!$processDebuggers && !$this->templateService->isEditingEnabled()) {
      return $build;
    }
    $items = [];
    foreach ($processDebuggers as $processDebugger) {
      if (!$processDebugger->isStarted()) {
        continue;
      }
      $hash = $processDebugger->getHistoryHash();
      $item = [
        '#theme' => 'top_bar_page_action',
        '#link' => [
          '#type' => 'link',
          '#title' => $processDebugger->getEventLabel(),
          '#url' => Url::fromRoute('entity.eca.edit_with.nextgen_modeler', ['eca' => $processDebugger->getEcaId()], [
            'query' => [
              'select' => $processDebugger->getEventId(),
              'hash' => $hash,
            ],
          ]),
          '#attributes' => [
            'data-modeler-eca-id' => $processDebugger->getEcaId(),
            'data-modeler-eca-event-id' => $processDebugger->getEventId(),
            'data-modeler-eca-hash' => $hash,
          ],
        ],
      ];
      (new Htmx())
        ->get($item['#link']['#url'])
        ->onlyMainContent()
        ->select('#nextgen-modeler-wrapper')
        ->target('.page-wrapper')
        ->swap('afterbegin')
        ->applyTo($item['#link']);
      $items[] = $item;
    }
    $build += [
      '#theme' => 'top_bar_page_actions',
      '#page_actions' => $items,
      '#featured_page_actions' => [
        'eca_template_editing' => [
          'page_action' => [
            '#theme' => 'top_bar_page_action',
            '#link' => [
              '#type' => 'link',
              '#title' => '',
              '#url' => '',
              '#attributes' => [
                'class' => ['eca-template-editing'],
                'title' => $this->t('Configure the site with ECA'),
              ],
            ],
          ],
          'icon' => [
            'pack_id' => 'eca_ng',
            'icon_id' => 'cog',
          ],
        ],
      ],
    ];
    if ($items) {
      $build['#attached']['library'][] = 'modeler/react-ui';
    }
    return $build;
  }

}
