<?php

declare(strict_types=1);

namespace Drupal\eca_quiz\Plugin\Action\AI;

use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai_integration_eca\Plugin\Action\AiConfigActionBase;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Form\FormStateInterface;
use Drupal\taxonomy\Entity\Term;
use Symfony\Component\Validator\Constraints\Optional;
use Symfony\Component\Validator\Constraints\Type;

/**
 * Describes the AI ai_integration_eca_execute_chat action with configurable trivia prompt.
 *
 * @Action(
 *   id = "eca_quiz_generate_trivia_response",
 *   label = @Translation("Generate Trivia Response"),
 *   description = @Translation("Generates trivia questions with configurable prompt placeholders via AI provider from the AI contrib module.")
 * )
 */
class GenerateTriviaResponseAction extends AiConfigActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
        'prompt' => 'Generate 10 trivia questions, each with 3 answer options, in JSON format. The questions must be highly distinct and not similar to one another. Each question must have exactly 3 answer options—no "Yes" or "No" answers. The index of answer options must start at 0 and end at 2. The correct answer must be unambiguous and must not overlap with the other options. Verify that the correct answer’s index is clear and logically fits the question before finalising it. The explanation must consist of two to three concise sentences. The category is {{ field_quiz_category }}, and the difficulty level is: {{ field_challenge_level }}. Ensure correct UK English spelling and grammar. The category must explicitly and exclusively be {{ field_quiz_category }}, and the questions must cover the following topic: {{ field_quiz_category }}. Do not include references or links to external content. Here is an example of the required structure: {"question": "What unusual job did Charlie Chaplin have?","options": ["Master painter", "Circus clown", "Actor"],"correctIndex": 2,"explanation": "Charlie Chaplin was an actor. He became world-famous for his silent film performances.","difficulty": "{{ field_challenge_level }}","categoryHumanReadable": "{{ field_quiz_category }}"}',
      ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt with placeholders'),
      '#default_value' => $this->configuration['prompt'],
      '#description' => $this->t('Without brackets.', ['@key' => 'prompt']),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    foreach (['categoryHumanReadable', 'difficulty', 'topicHumanReadable', 'prompt'] as $key) {
      $this->configuration[$key] = $form_state->getValue($key);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $entity = $this->tokenService->getTokenData('entity');

    if ($entity === null) {
      return;
    }

    $challenge_level = $entity->field_challenge_level->value;
    $quiz_category = (function () use ($entity) {
      $term = Term::load($entity->field_quiz_category->first()->getValue()['target_id']);
      return $term->label();
    })();

    $modelData = $this->getModelData();
    try {
      $provider = $this->loadModelProvider();
    }
    catch (PluginException $e) {
      return;
    }

    $replacements = [
      '{{ field_quiz_category }}' => $quiz_category,
      '{{ field_challenge_level }}' => $challenge_level,
    ];

    $prompt = strtr($this->configuration['prompt'], $replacements);

    $data = [];
    if (!empty($this->configuration['token_input']) && $this->tokenService->hasTokenData($this->configuration['token_input'])) {
      $token_value = $this->tokenService->getTokenData($this->configuration['token_input']);
      $data = [
        $this->configuration['token_input'] => $token_value,
      ];
    }

    $finalPrompt = $this->tokenService->replace($prompt, $data);

    $chatInput = [
      new ChatMessage('user', $finalPrompt),
    ];

    $modelConfig = $this->getModelConfig();
    if (!empty($modelConfig['system_name']) && !empty($modelConfig['system_prompt'])) {
      $chatInput[] = new ChatMessage($modelConfig['system_name'], $modelConfig['system_prompt']);
      unset($modelConfig['system_name'], $modelConfig['system_prompt']);
    }

    $messages = new ChatInput($chatInput);
    $provider->setConfiguration($this->getModelConfig());
    $message = $provider->chat($messages, $modelData['model_id'])->getNormalized();
    $response = trim($message->getText());
    if ($response === '') {
      $response = $this->t('No result could be generated.');
    }

    $this->tokenService->addTokenData($this->configuration['token_result'], $response);
  }

  /**
   * {@inheritdoc}
   */
  protected function getOperationType(): string {
    return 'chat';
  }

  /**
   * {@inheritdoc}
   */
  protected function getExtraConstraints(): array {
    return [
      'system_name' => new Optional([
        'constraints' => [new Type('string')],
      ]),
      'system_prompt' => new Optional([
        'constraints' => [new Type('string')],
      ]),
    ];
  }

}
