<?php

declare(strict_types=1);

namespace Drupal\eca_quiz\Plugin\Action\Entity;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Action\ConfigurableActionBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\quiz\Entity\QuizQuestion;
use Drupal\token\TokenInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

#[Action(
  id: 'create_quiz_question_action',
  label: new TranslatableMarkup('Create quiz question action'),
  type: 'quiz',
)]
class CreateQuizQuestionAction extends ConfigurableActionBase implements ContainerFactoryPluginInterface {

  /**
   * Constructor for the CustomCreateQuizQuestionAction.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager service.
   * @param \Drupal\token\TokenInterface $token
   *   The token service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entityTypeManager,
    private readonly TokenInterface $token,
    private readonly LoggerInterface $logger
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('token'),
      $container->get('logger.channel.eca_quiz'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($object = NULL) {
    if ($this->token->hasTokenData($this->configuration['prompt_result_token_name'])) {
      $prompt_result_token_name = $this->token->getTokenData($this->configuration['prompt_result_token_name']);
      $prompt_result = $prompt_result_token_name->getValue();

      $questions_data = json_decode($prompt_result, TRUE);

      if (is_array($questions_data)) {
        try {
          foreach ($questions_data as $question_data) {
            $alternatives = [];
            foreach ($question_data['options'] as $index => $option) {
              $is_correct = ($index === $question_data['correctIndex']);
              $paragraph = Paragraph::create([
                'type' => 'multichoice',
                'multichoice_correct' => $is_correct ? 1 : 0,
                'multichoice_answer' => $option,
                'multichoice_feedback_chosen' => $is_correct ? 'Correct! ' . $question_data['explanation'] : 'Incorrect. ' . $question_data['explanation'],
                'multichoice_feedback_not_chosen' => $is_correct ? 'You missed the correct answer.' : '',
                'multichoice_score_chosen' => $is_correct ? 1 : -1,
                'multichoice_score_not_chosen' => 0,
              ]);
              $paragraph->save();
              $alternatives[] = $paragraph;
            }

            // Create the quiz question.
            $question = QuizQuestion::create([
              'title' => $question_data['question'],
              'type' => 'multichoice',
              'choice_multi' => 0,
              'choice_random' => 0,
              'choice_boolean' => 0,
              'body' => $question_data['question'],
              'field_category' => [
                (function() use ($question_data) {
                  $term_label = $question_data['categoryHumanReadable'];
                  $terms = \Drupal::entityTypeManager()
                    ->getStorage('taxonomy_term')
                    ->loadByProperties([
                      'name' => $term_label,
                      'vid' => 'quiz_category',
                    ]);
                  $term = reset($terms);
                  return $term ? ['target_id' => $term->id()] : NULL;
                })(),
              ],
            ]);

            // Attach alternatives to the question.
            foreach ($alternatives as $alternative) {
              $question->get('alternatives')->appendItem($alternative);
            }

            $question->save();
          }
        } catch (\Throwable $e) {
          $this->logger->error($e->getMessage());
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['prompt_result_token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name of the prompt result'),
      '#default_value' => $this->configuration['prompt_result_token_name'] ?? '',
      '#description' => $this->t('Provide the name of the prompt result.'),
      '#weight' => 1,
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return [
        'prompt_result_token_name' => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    $this->configuration['prompt_result_token_name'] = $form_state->getValue('prompt_result_token_name');
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    return $return_as_object ? AccessResult::allowed() : TRUE;
  }

}
