<?php

declare(strict_types=1);

namespace Drupal\eca_quiz\Plugin\Action\AI;

use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai_integration_eca\Plugin\Action\AiConfigActionBase;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\Validator\Constraints\Optional;
use Symfony\Component\Validator\Constraints\Type;

/**
 * Describes the AI ai_integration_eca_execute_chat action with configurable trivia prompt.
 *
 * @Action(
 *   id = "eca_quiz_generate_trivia_response",
 *   label = @Translation("Generate Trivia Response"),
 *   description = @Translation("Generates trivia questions with configurable prompt placeholders via AI provider from the AI contrib module.")
 * )
 */
class GenerateTriviaResponseAction extends AiConfigActionBase {

  use StringTranslationTrait;

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return parent::defaultConfiguration() + [
        'prompt' => $this->t('Generate 10 trivia questions, each with 3 answer options, in JSON format. The questions must be highly distinct and not similar to one another. Each question must have exactly 3 answer options—no "Yes" or "No" answers. The index of answer options must start at 0 and end at 2. The correct answer must be unambiguous and must not overlap with the other options. Verify that the correct answer’s index is clear and logically fits the question before finalising it. The explanation must consist of two to three concise sentences. The label is [node:label], and the difficulty level is: [node:field_challenge_level]. Ensure correct UK English spelling and grammar. The label must explicitly and exclusively be [node:label], and the questions must cover the following topic: [node:label]. Do not include references or links to external content. Here is an example of the required structure: {"question": "What unusual job did Charlie Chaplin have?","options": ["Master painter", "Circus clown", "Actor"],"correctIndex": 2,"explanation": "Charlie Chaplin was an actor. He became world-famous for his silent film performances.","difficulty": "[node:field_challenge_level]","term_id": "[node:field_quiz_category:target_id]"}'),
      ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Prompt with placeholders'),
      '#default_value' => $this->configuration['prompt'],
      '#description' => $this->t('Without brackets.', ['@key' => 'prompt']),
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    parent::submitConfigurationForm($form, $form_state);

    foreach (['categoryHumanReadable', 'difficulty', 'topicHumanReadable', 'prompt'] as $key) {
      $this->configuration[$key] = $form_state->getValue($key);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $entity = $this->tokenService->getTokenData('entity');

    if ($entity === null) {
      return;
    }

    $modelData = $this->getModelData();
    try {
      $provider = $this->loadModelProvider();
    }
    catch (PluginException $e) {
      return;
    }

    $finalPrompt = $this->tokenService->replace(
      $this->configuration['prompt'],
      [
        'node' => $this->tokenService->getTokenData('entity')
      ]
    );

    $chatInput = [
      new ChatMessage('user', $finalPrompt),
    ];

    $modelConfig = $this->getModelConfig();
    if (!empty($modelConfig['system_name']) && !empty($modelConfig['system_prompt'])) {
      $chatInput[] = new ChatMessage($modelConfig['system_name'], $modelConfig['system_prompt']);
      unset($modelConfig['system_name'], $modelConfig['system_prompt']);
    }

    $messages = new ChatInput($chatInput);
    $provider->setConfiguration($this->getModelConfig());
    $message = $provider->chat($messages, $modelData['model_id'])->getNormalized();
    $response = trim($message->getText());
    if ($response === '') {
      $response = $this->t('No result could be generated.');
    }

    $this->tokenService->addTokenData($this->configuration['token_result'], $response);
  }

  /**
   * {@inheritdoc}
   */
  protected function getOperationType(): string {
    return 'chat';
  }

  /**
   * {@inheritdoc}
   */
  protected function getExtraConstraints(): array {
    return [
      'system_name' => new Optional([
        'constraints' => [new Type('string')],
      ]),
      'system_prompt' => new Optional([
        'constraints' => [new Type('string')],
      ]),
    ];
  }

}
