<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_zoom\Service\ZoomService;

/**
 * Create a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_create_meeting',
  label: new TranslatableMarkup('Zoom: Create Meeting'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Schedule new Zoom meetings with participants and configuration.'),
  version_introduced: '1.0.0',
)]
final class CreateMeeting extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'user_id' => ZoomService::DEFAULT_USER_ID,
      'topic' => '',
      'start_time' => '',
      'duration' => (string) ZoomService::DEFAULT_DURATION,
      'timezone' => '',
      'additional_settings' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addUserIdConfigurationForm($form, $form_state);

    $form['topic'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Meeting Topic'),
      '#description' => $this->t('The topic or title of the meeting.'),
      '#default_value' => $this->configuration['topic'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['start_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time'),
      '#description' => $this->t('Start time in ISO 8601 format (e.g., 2023-12-01T10:00:00Z). Leave empty for instant meeting.'),
      '#default_value' => $this->configuration['start_time'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['duration'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Duration (minutes)'),
      '#description' => $this->t('Meeting duration in minutes.'),
      '#default_value' => $this->configuration['duration'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['timezone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Timezone'),
      '#description' => $this->t('Timezone for the meeting (e.g., America/New_York). Leave empty to use account default.'),
      '#default_value' => $this->configuration['timezone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['additional_settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Settings (YAML)'),
      '#description' => $this->t('Additional meeting settings in YAML format. See <a href="@url" target="_blank">Zoom API documentation</a> for available options. Example:<br><code>password: "secret123"<br>alternative_hosts: "user@example.com"<br>settings:<br>&nbsp;&nbsp;waiting_room: true<br>&nbsp;&nbsp;host_video: true<br>&nbsp;&nbsp;participant_video: false<br>&nbsp;&nbsp;mute_upon_entry: true<br>recurrence:<br>&nbsp;&nbsp;type: 2<br>&nbsp;&nbsp;end_times: 7</code>', [
        '@url' => 'https://developers.zoom.us/docs/api/rest/reference/zoom-api/methods/#operation/meetingCreate',
      ]),
      '#default_value' => $this->configuration['additional_settings'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the created meeting data, including meeting_id, join_url, host_url, and password.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }


  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $user_id = $this->getUserId();
    $topic = $this->tokenService->replacePlain($this->configuration['topic']);
    $start_time = $this->tokenService->replacePlain($this->configuration['start_time']);
    $duration = (int) $this->tokenService->replacePlain($this->configuration['duration']);
    $timezone = $this->tokenService->replacePlain($this->configuration['timezone']);

    // Build base meeting/webinar data with essential fields
    // Set default type based on resource type and whether start_time is provided
    $default_type = $resource_type === ZoomService::RESOURCE_TYPE_WEBINAR ? 5 : (empty($start_time) ? 1 : 2);

    $meeting_data = [
      'topic' => $topic,
      'type' => $default_type,
      'duration' => $duration,
    ];

    // Add formatted start time if provided
    if (!empty($start_time)) {
      $formatted_start_time = $this->zoomService->formatDateTimeForApi($start_time, $timezone);
      if ($formatted_start_time) {
        $meeting_data['start_time'] = $formatted_start_time;
      }
    }

    // Add timezone if provided
    if (!empty($timezone)) {
      $meeting_data['timezone'] = $timezone;
    }

    // Parse and merge additional YAML settings
    $meeting_data = $this->mergeYamlSettings($meeting_data, 'additional_settings');

    $response = $this->zoomService->createMeeting($user_id, $meeting_data, $resource_type);

    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

}
