<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Delete a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_delete_meeting',
  label: new TranslatableMarkup('Zoom: Delete Meeting'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Cancel scheduled meetings and notify participants.'),
  version_introduced: '1.0.0',
)]
final class DeleteMeeting extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'send_notification' => '1',
      'cancellation_reason' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state);

    $form['send_notification'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Send Notification'),
      '#description' => $this->t('Set to "1", "true", or "yes" to send cancellation notification to registered participants.'),
      '#default_value' => $this->configuration['send_notification'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['cancellation_reason'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Cancellation Reason'),
      '#description' => $this->t('Optional reason for meeting cancellation that will be included in the notification.'),
      '#default_value' => $this->configuration['cancellation_reason'],
      '#rows' => 3,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the deletion result data.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $send_notification = $this->tokenService->replaceClear($this->configuration['send_notification']);
    $cancellation_reason = $this->tokenService->replacePlain($this->configuration['cancellation_reason']);

    $success = $this->zoomService->deleteMeeting($meeting_id, !empty($send_notification), $cancellation_reason, $resource_type);

    if (!empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $result_data = [
        'meeting_id' => $meeting_id,
        'deleted' => $success,
        'notification_sent' => !empty($send_notification),
        'cancellation_reason' => $cancellation_reason,
        'timestamp' => date('c'),
      ];

      $this->tokenService->addTokenData($token_name, $result_data);
    }
  }

}
