<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_zoom\Service\ZoomService;

/**
 * Get a Zoom meeting participant report.
 */
#[Action(
  id: 'eca_zoom_get_participant_report',
  label: new TranslatableMarkup('Zoom: Get Participant Report'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve detailed individual participant engagement and activity data.'),
  version_introduced: '1.0.0',
)]
final class GetParticipantReport extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'page_size' => (string) ZoomService::DEFAULT_PAGE_SIZE,
      'next_page_token' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state, TRUE);

    $form['page_size'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page Size'),
      '#description' => $this->t('Number of participants to retrieve per page (1-300).'),
      '#default_value' => $this->configuration['page_size'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['next_page_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Next Page Token'),
      '#description' => $this->t('Token for retrieving the next page of results. Leave empty for first page.'),
      '#default_value' => $this->configuration['next_page_token'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the participant report data including engagement scores and session details.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $page_size = (int) $this->tokenService->replacePlain($this->configuration['page_size']);
    $next_page_token = $this->tokenService->replacePlain($this->configuration['next_page_token']);

    $response = $this->zoomService->getParticipantReport($meeting_id, $page_size, $next_page_token, $resource_type);

    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

}
