<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_zoom\Service\ZoomService;

/**
 * List user meetings.
 */
#[Action(
  id: 'eca_zoom_list_meetings',
  label: new TranslatableMarkup('Zoom: List User Meetings'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve meetings for a specific user with filtering.'),
  version_introduced: '1.0.0',
)]
final class ListMeetings extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'user_id' => ZoomService::DEFAULT_USER_ID,
      'meeting_type' => ZoomService::MEETING_TYPE_SCHEDULED,
      'page_size' => (string) ZoomService::DEFAULT_PAGE_SIZE,
      'next_page_token' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addUserIdConfigurationForm($form, $form_state);

    $form['meeting_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Meeting Type'),
      '#description' => $this->t('Type of meetings to retrieve.'),
      '#options' => [
        ZoomService::MEETING_TYPE_SCHEDULED => $this->t('Scheduled meetings'),
        ZoomService::MEETING_TYPE_LIVE => $this->t('Live meetings'),
        ZoomService::MEETING_TYPE_UPCOMING => $this->t('Upcoming meetings'),
        ZoomService::MEETING_TYPE_PREVIOUS => $this->t('Previous meetings'),
      ],
      '#default_value' => $this->configuration['meeting_type'],
      '#required' => TRUE,
    ];

    $form['page_size'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page Size'),
      '#description' => $this->t('Number of meetings to retrieve per page (1-300).'),
      '#default_value' => $this->configuration['page_size'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['next_page_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Next Page Token'),
      '#description' => $this->t('Token for retrieving the next page of results. Leave empty for first page.'),
      '#default_value' => $this->configuration['next_page_token'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the meetings list data, including next_page_token for pagination.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $user_id = $this->getUserId();
    $meeting_type = $this->configuration['meeting_type'];
    $page_size = (int) $this->tokenService->replacePlain($this->configuration['page_size']);
    $next_page_token = $this->tokenService->replacePlain($this->configuration['next_page_token']);

    $response = $this->zoomService->listMeetings($user_id, $meeting_type, $page_size, $next_page_token, $resource_type);

    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

}
