<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Form\FormStateInterface;
use Drupal\eca\Plugin\Action\ConfigurableActionBase;
use Drupal\eca_zoom\Service\ZoomService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for Zoom ECA actions.
 */
abstract class ZoomActionBase extends ConfigurableActionBase {

  /**
   * The Zoom service.
   */
  protected ZoomService $zoomService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->zoomService = $container->get('eca_zoom.zoom');
    return $instance;
  }

  /**
   * Gets default configuration for resource type.
   *
   * @return array
   *   Default configuration array.
   */
  protected function getResourceTypeDefaultConfig(): array {
    return [
      'resource_type' => ZoomService::RESOURCE_TYPE_MEETING,
      'resource_type_token' => '',
    ];
  }

  /**
   * Adds resource type configuration form elements.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array with resource type fields.
   */
  protected function addResourceTypeConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['resource_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Meeting Type'),
      '#description' => $this->t('Whether this is a meeting or webinar.'),
      '#options' => [
        ZoomService::RESOURCE_TYPE_MEETING => $this->t('Meeting'),
        ZoomService::RESOURCE_TYPE_WEBINAR => $this->t('Webinar'),
        ZoomService::RESOURCE_TYPE_FROM_TOKEN => $this->t('From Token'),
      ],
      '#default_value' => $this->configuration['resource_type'] ?? ZoomService::RESOURCE_TYPE_MEETING,
      '#required' => TRUE,
      '#weight' => -100,
    ];

    $form['resource_type_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Meeting Type Token'),
      '#description' => $this->t('Token value that evaluates to "meeting" or "webinar".'),
      '#default_value' => $this->configuration['resource_type_token'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="resource_type"]' => ['value' => ZoomService::RESOURCE_TYPE_FROM_TOKEN],
        ],
        'required' => [
          ':input[name="resource_type"]' => ['value' => ZoomService::RESOURCE_TYPE_FROM_TOKEN],
        ],
      ],
      '#eca_token_replacement' => TRUE,
      '#weight' => -99,
    ];

    return $form;
  }

  /**
   * Determines the actual resource type from configuration.
   *
   * @return string
   *   Either 'meeting' or 'webinar'.
   */
  protected function getResourceType(): string {
    $resource_type = $this->configuration['resource_type'] ?? ZoomService::RESOURCE_TYPE_MEETING;

    if ($resource_type === ZoomService::RESOURCE_TYPE_FROM_TOKEN) {
      $token_value = strtolower(trim($this->tokenService->replacePlain($this->configuration['resource_type_token'] ?? '')));
      $resource_type = ($token_value === ZoomService::RESOURCE_TYPE_WEBINAR) ? ZoomService::RESOURCE_TYPE_WEBINAR : ZoomService::RESOURCE_TYPE_MEETING;
    }

    return $resource_type;
  }

  /**
   * Adds meeting ID configuration form element.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param bool $allow_uuid
   *   Whether to allow UUID in the description.
   *
   * @return array
   *   The form array with meeting_id field.
   */
  protected function addMeetingIdConfigurationForm(array $form, FormStateInterface $form_state, bool $allow_uuid = FALSE): array {
    $description = $allow_uuid
      ? $this->t('The meeting ID, UUID, or full Zoom meeting URL. Use the meeting UUID for past meetings. If a URL is provided, the meeting ID will be extracted automatically.')
      : $this->t('The meeting ID or full Zoom meeting URL. If a URL is provided, the meeting ID will be extracted automatically.');

    $form['meeting_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Meeting ID or URL'),
      '#description' => $description,
      '#default_value' => $this->configuration['meeting_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * Gets the meeting ID from configuration with token replacement and URL extraction.
   *
   * @return string
   *   The extracted meeting ID.
   */
  protected function getMeetingId(): string {
    $meeting_id_or_url = $this->tokenService->replacePlain($this->configuration['meeting_id']);
    return $this->zoomService->extractMeetingId($meeting_id_or_url);
  }

  /**
   * Adds user ID configuration form element.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The form array with user_id field.
   */
  protected function addUserIdConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form['user_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('User ID'),
      '#description' => $this->t('The user ID or email address of the meeting host. Use "me" for the authenticated user.'),
      '#default_value' => $this->configuration['user_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    return $form;
  }

  /**
   * Gets the user ID from configuration with token replacement.
   *
   * @return string
   *   The user ID.
   */
  protected function getUserId(): string {
    return $this->tokenService->replacePlain($this->configuration['user_id']);
  }

  /**
   * Parses YAML settings and merges them with base data.
   *
   * @param array $base_data
   *   The base data array to merge into.
   * @param string $config_key
   *   The configuration key containing the YAML string.
   *
   * @return array
   *   The merged data array.
   */
  protected function mergeYamlSettings(array $base_data, string $config_key): array {
    $yaml_string = $this->tokenService->replacePlain($this->configuration[$config_key] ?? '');

    if (empty($yaml_string)) {
      return $base_data;
    }

    try {
      $yaml_data = \Symfony\Component\Yaml\Yaml::parse($yaml_string);
      if (is_array($yaml_data)) {
        return $this->mergeArraysRecursive($base_data, $yaml_data);
      }
    }
    catch (\Exception $e) {
      // Log YAML parsing error but continue with basic settings
      if (method_exists($this, 'logger') && isset($this->logger)) {
        $this->logger->error('Failed to parse YAML settings from @key: @message', [
          '@key' => $config_key,
          '@message' => $e->getMessage(),
        ]);
      }
    }

    return $base_data;
  }

  /**
   * Recursively merge arrays, preserving nested structures.
   *
   * @param array $array1
   *   The base array.
   * @param array $array2
   *   The array to merge in.
   *
   * @return array
   *   The merged array.
   */
  protected function mergeArraysRecursive(array $array1, array $array2): array {
    foreach ($array2 as $key => $value) {
      if (is_array($value) && isset($array1[$key]) && is_array($array1[$key])) {
        $array1[$key] = $this->mergeArraysRecursive($array1[$key], $value);
      }
      else {
        $array1[$key] = $value;
      }
    }
    return $array1;
  }

}
