<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Add a registrant to a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_add_registrant',
  label: new TranslatableMarkup('Zoom: Add Meeting Registrant'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Register users for meetings and webinars.'),
  version_introduced: '1.0.0',
)]
final class AddRegistrant extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'email' => '',
      'first_name' => '',
      'last_name' => '',
      'additional_settings' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state);

    $form['email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Email Address'),
      '#description' => $this->t('The email address of the registrant.'),
      '#default_value' => $this->configuration['email'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['first_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('First Name'),
      '#description' => $this->t('The first name of the registrant.'),
      '#default_value' => $this->configuration['first_name'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['last_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Last Name'),
      '#description' => $this->t('The last name of the registrant.'),
      '#default_value' => $this->configuration['last_name'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['additional_settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Settings (YAML)'),
      '#description' => $this->t('Additional registration settings in YAML format. See <a href="@url" target="_blank">Zoom API documentation</a> for available options. Example:<br><code>auto_approve: true<br>custom_questions:<br>&nbsp;&nbsp;- title: "Dietary restrictions"<br>&nbsp;&nbsp;&nbsp;&nbsp;value: "None"</code>', [
        '@url' => 'https://developers.zoom.us/docs/api/rest/reference/zoom-api/methods/#operation/meetingRegistrantCreate',
      ]),
      '#default_value' => $this->configuration['additional_settings'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the registration data including registrant ID and join URL.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $email = $this->tokenService->replacePlain($this->configuration['email']);
    $first_name = $this->tokenService->replacePlain($this->configuration['first_name']);
    $last_name = $this->tokenService->replacePlain($this->configuration['last_name']);

    // Build base registrant data
    $registrant_data = [
      'email' => $email,
      'first_name' => $first_name,
    ];

    if (!empty($last_name)) {
      $registrant_data['last_name'] = $last_name;
    }

    // Parse and merge additional YAML settings
    $registrant_data = $this->mergeYamlSettings($registrant_data, 'additional_settings');

    $response = $this->zoomService->addRegistrant($meeting_id, $registrant_data, $resource_type);

    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

}
