<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Get a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_get_meeting',
  label: new TranslatableMarkup('Zoom: Get Meeting'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve meeting details and settings.'),
  version_introduced: '1.0.0',
)]
final class GetMeeting extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state);

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the meeting data including all settings. Use this token to read current values before updating.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $response = $this->zoomService->getMeeting($meeting_id, $resource_type);

    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

}
