<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;
use Drupal\eca_zoom\Service\ZoomService;

/**
 * List registrants for a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_list_registrants',
  label: new TranslatableMarkup('Zoom: List Meeting Registrants'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Retrieve list of registered participants for a meeting.'),
  version_introduced: '1.0.0',
)]
final class ListRegistrants extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'status' => ZoomService::REGISTRATION_STATUS_APPROVED,
      'page_size' => (string) ZoomService::DEFAULT_PAGE_SIZE,
      'next_page_token' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state);

    $form['status'] = [
      '#type' => 'select',
      '#title' => $this->t('Registration Status'),
      '#description' => $this->t('Filter registrants by their registration status.'),
      '#options' => [
        ZoomService::REGISTRATION_STATUS_APPROVED => $this->t('Approved'),
        ZoomService::REGISTRATION_STATUS_PENDING => $this->t('Pending approval'),
        ZoomService::REGISTRATION_STATUS_DENIED => $this->t('Denied'),
      ],
      '#default_value' => $this->configuration['status'],
      '#required' => TRUE,
    ];

    $form['page_size'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page Size'),
      '#description' => $this->t('Number of registrants to retrieve per page (1-300).'),
      '#default_value' => $this->configuration['page_size'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['next_page_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Next Page Token'),
      '#description' => $this->t('Token for retrieving the next page of results. Leave empty for first page.'),
      '#default_value' => $this->configuration['next_page_token'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the registrants list data, including participant details and next_page_token for pagination.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $status = $this->configuration['status'];
    $page_size = (int) $this->tokenService->replacePlain($this->configuration['page_size']);
    $next_page_token = $this->tokenService->replacePlain($this->configuration['next_page_token']);

    $response = $this->zoomService->listRegistrants($meeting_id, $status, $page_size, $next_page_token, $resource_type);

    if ($response && !empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $this->tokenService->addTokenData($token_name, $response);
    }
  }

}
