<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Remove a registrant from a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_remove_registrant',
  label: new TranslatableMarkup('Zoom: Remove Meeting Registrant'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Unregister registrants from meetings and webinars.'),
  version_introduced: '1.0.0',
)]
final class RemoveRegistrant extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'registrant_id' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state);

    $form['registrant_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Registrant ID'),
      '#description' => $this->t('The registrant ID to remove. This is typically returned when adding a registrant.'),
      '#default_value' => $this->configuration['registrant_id'],
      '#required' => TRUE,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the removal result data.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $registrant_id = $this->tokenService->replacePlain($this->configuration['registrant_id']);

    $success = $this->zoomService->removeRegistrant($meeting_id, $registrant_id, $resource_type);

    if (!empty($this->configuration['token_name'])) {
      $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
      $result_data = [
        'meeting_id' => $meeting_id,
        'registrant_id' => $registrant_id,
        'removed' => $success,
        'timestamp' => date('c'),
      ];

      $this->tokenService->addTokenData($token_name, $result_data);
    }
  }

}
