<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Plugin\Action;

use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\eca\Attribute\EcaAction;

/**
 * Update a Zoom meeting.
 */
#[Action(
  id: 'eca_zoom_update_meeting',
  label: new TranslatableMarkup('Zoom: Update Meeting'),
  category: new TranslatableMarkup('Zoom'),
  type: 'system',
)]
#[EcaAction(
  description: new TranslatableMarkup('Modify existing meeting details and settings.'),
  version_introduced: '1.0.0',
)]
final class UpdateMeeting extends ZoomActionBase {

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration(): array {
    return $this->getResourceTypeDefaultConfig() + [
      'meeting_id' => '',
      'topic' => '',
      'start_time' => '',
      'duration' => '',
      'timezone' => '',
      'additional_settings' => '',
      'token_name' => '',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form = $this->addResourceTypeConfigurationForm($form, $form_state);
    $form = $this->addMeetingIdConfigurationForm($form, $form_state);

    $form['topic'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Meeting Topic'),
      '#description' => $this->t('The topic or title of the meeting. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['topic'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['start_time'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Start Time'),
      '#description' => $this->t('Start time in ISO 8601 format. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['start_time'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['duration'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Duration (minutes)'),
      '#description' => $this->t('Meeting duration in minutes. Leave empty to keep current value.'),
      '#default_value' => $this->configuration['duration'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['timezone'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Timezone'),
      '#description' => $this->t('Timezone for the meeting (e.g., America/New_York). Leave empty to keep current value.'),
      '#default_value' => $this->configuration['timezone'],
      '#eca_token_replacement' => TRUE,
    ];

    $form['additional_settings'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Additional Settings (YAML)'),
      '#description' => $this->t('Additional meeting settings to update in YAML format. See <a href="@url" target="_blank">Zoom API documentation</a> for available options. Example:<br><code>password: "newsecret"<br>settings:<br>&nbsp;&nbsp;waiting_room: false<br>&nbsp;&nbsp;host_video: true</code>', [
        '@url' => 'https://developers.zoom.us/docs/api/rest/reference/zoom-api/methods/#operation/meetingUpdate',
      ]),
      '#default_value' => $this->configuration['additional_settings'],
      '#rows' => 8,
      '#eca_token_replacement' => TRUE,
    ];

    $form['token_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Token name'),
      '#description' => $this->t('Provide the name of a token that holds the updated meeting data.'),
      '#default_value' => $this->configuration['token_name'],
      '#eca_token_replacement' => TRUE,
      '#weight' => 50,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function execute(): void {
    $resource_type = $this->getResourceType();
    $meeting_id = $this->getMeetingId();

    $topic = $this->tokenService->replacePlain($this->configuration['topic']);
    $start_time = $this->tokenService->replacePlain($this->configuration['start_time']);
    $duration = $this->tokenService->replacePlain($this->configuration['duration']);
    $timezone = $this->tokenService->replacePlain($this->configuration['timezone']);

    // Build meeting data with only non-empty values
    $meeting_data = [];

    if (!empty($topic)) {
      $meeting_data['topic'] = $topic;
    }

    if (!empty($start_time)) {
      $formatted_start_time = $this->zoomService->formatDateTimeForApi($start_time, $timezone);
      if ($formatted_start_time) {
        $meeting_data['start_time'] = $formatted_start_time;
      }
    }

    if (!empty($duration)) {
      $meeting_data['duration'] = (int) $duration;
    }

    if (!empty($timezone)) {
      $meeting_data['timezone'] = $timezone;
    }

    // Parse and merge additional YAML settings
    $meeting_data = $this->mergeYamlSettings($meeting_data, 'additional_settings');

    // Only call API if we have data to update
    if (!empty($meeting_data)) {
      $response = $this->zoomService->updateMeeting($meeting_id, $meeting_data, $resource_type);

      if ($response && !empty($this->configuration['token_name'])) {
        $token_name = $this->tokenService->replacePlain($this->configuration['token_name']);
        $this->tokenService->addTokenData($token_name, $response);
      }
    }
  }

}
