<?php

declare(strict_types=1);

namespace Drupal\eca_zoom\Service;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Utility\Token;
use Drupal\eca_zoom\DateTimeFormattingTrait;
use Drupal\zoomapi\Plugin\ApiTools\Client;

/**
 * Service for Zoom API operations.
 */
class ZoomService {

  use DateTimeFormattingTrait;

  /**
   * Resource type: Meeting.
   */
  const string RESOURCE_TYPE_MEETING = 'meeting';

  /**
   * Resource type: Webinar.
   */
  const string RESOURCE_TYPE_WEBINAR = 'webinar';

  /**
   * Resource type from token
   */
  const string RESOURCE_TYPE_FROM_TOKEN = 'token';

  /**
   * Meeting type: Scheduled meetings.
   */
  const string MEETING_TYPE_SCHEDULED = 'scheduled';

  /**
   * Meeting type: Live meetings.
   */
  const string MEETING_TYPE_LIVE = 'live';

  /**
   * Meeting type: Upcoming meetings.
   */
  const string MEETING_TYPE_UPCOMING = 'upcoming';

  /**
   * Meeting type: Previous meetings.
   */
  const string MEETING_TYPE_PREVIOUS = 'previous';

  /**
   * Registration status: Approved.
   */
  const string REGISTRATION_STATUS_APPROVED = 'approved';

  /**
   * Registration status: Pending approval.
   */
  const string REGISTRATION_STATUS_PENDING = 'pending';

  /**
   * Registration status: Denied.
   */
  const string REGISTRATION_STATUS_DENIED = 'denied';

  /**
   * Default user ID representing the authenticated user.
   */
  const string DEFAULT_USER_ID = 'me';

  /**
   * Default meeting duration in minutes.
   */
  const int DEFAULT_DURATION = 60;

  /**
   * Default page size for paginated results.
   */
  const int DEFAULT_PAGE_SIZE = 30;

  /**
   * The Zoom API client.
   *
   * @var \Drupal\zoomapi\Plugin\ApiTools\Client
   */
  protected Client $zoomClient;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * The token service.
   *
   * @var \Drupal\Core\Utility\Token
   */
  protected Token $token;

  /**
   * Constructs a ZoomService object.
   *
   * @param \Drupal\zoomapi\Plugin\ApiTools\Client $zoom_client
   *   The Zoom API client.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\Core\Utility\Token $token
   *   The token service.
   */
  public function __construct(Client $zoom_client, LoggerChannelInterface $logger, Token $token) {
    $this->zoomClient = $zoom_client;
    $this->logger = $logger;
    $this->token = $token;
  }

  /**
   * Creates a Zoom meeting or webinar.
   *
   * @param string $user_id
   *   The user ID (email or user ID).
   * @param array $meeting_data
   *   The meeting/webinar data.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The meeting/webinar response data or NULL on failure.
   */
  public function createMeeting(string $user_id, array $meeting_data, string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "users/{$user_id}/webinars" : "users/{$user_id}/meetings";

      $response = $this->zoomClient->post($endpoint, [
        'json' => $meeting_data,
      ]);

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to create Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Extracts meeting ID from a URL or returns the ID as-is.
   *
   * @param string $meeting_id_or_url
   *   The meeting ID or full Zoom URL.
   *
   * @return string
   *   The extracted meeting ID.
   */
  public function extractMeetingId(string $meeting_id_or_url): string {
    // If it looks like a URL, extract the ID
    if (preg_match('#(?:https?://)?(?:[\w-]+\.)?zoom\.us/[a-z]+/(\d+)#i', $meeting_id_or_url, $matches)) {
      return $matches[1];
    }

    // Otherwise return as-is (already an ID)
    return $meeting_id_or_url;
  }

  /**
   * Gets a Zoom meeting or webinar.
   *
   * @param string $meeting_id
   *   The meeting/webinar ID.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The meeting/webinar data or NULL on failure.
   */
  public function getMeeting(string $meeting_id, string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "webinars/{$meeting_id}" : "meetings/{$meeting_id}";

      return $this->zoomClient->get($endpoint);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Updates a Zoom meeting or webinar.
   *
   * @param string $meeting_id
   *   The meeting/webinar ID.
   * @param array $meeting_data
   *   The meeting/webinar data to update.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The updated meeting/webinar data or NULL on failure.
   */
  public function updateMeeting(string $meeting_id, array $meeting_data, string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "webinars/{$meeting_id}" : "meetings/{$meeting_id}";

      return $this->zoomClient->patch($endpoint, [
        'json' => $meeting_data,
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to update Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Deletes a Zoom meeting or webinar.
   *
   * @param string $meeting_id
   *   The meeting/webinar ID.
   * @param bool $send_notification
   *   Whether to send notification to participants.
   * @param string $cancellation_reason
   *   The cancellation reason.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function deleteMeeting(string $meeting_id, bool $send_notification = TRUE, string $cancellation_reason = '', string $resource_type = self::RESOURCE_TYPE_MEETING): bool {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "webinars/{$meeting_id}" : "meetings/{$meeting_id}";

      $query_params = [];
      if ($send_notification) {
        $query_params['schedule_for_reminder'] = 'true';
      }
      if (!empty($cancellation_reason)) {
        $query_params['cancel_meeting_reminder'] = $cancellation_reason;
      }

      $this->zoomClient->delete($endpoint, [
        'query' => $query_params,
      ]);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to delete Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Lists meetings for a user.
   *
   * @param string $user_id
   *   The user ID.
   * @param string $type
   *   The meeting type (scheduled, live, upcoming, previous).
   * @param int $page_size
   *   The page size.
   * @param string $next_page_token
   *   The next page token.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The meetings list or NULL on failure.
   */
  public function listMeetings(string $user_id, string $type = self::MEETING_TYPE_SCHEDULED, int $page_size = self::DEFAULT_PAGE_SIZE, string $next_page_token = '', string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "users/{$user_id}/webinars" : "users/{$user_id}/meetings";

      $query_params = [
        'type' => $type,
        'page_size' => $page_size,
      ];
      if (!empty($next_page_token)) {
        $query_params['next_page_token'] = $next_page_token;
      }

      return $this->zoomClient->get($endpoint, [
        'query' => $query_params,
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Adds a registrant to a meeting.
   *
   * @param string $meeting_id
   *   The meeting ID.
   * @param array $registrant_data
   *   The registrant data.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The registration response or NULL on failure.
   */
  public function addRegistrant(string $meeting_id, array $registrant_data, string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "webinars/{$meeting_id}/registrants" : "meetings/{$meeting_id}/registrants";

      $response = $this->zoomClient->post($endpoint, [
        'json' => $registrant_data,
      ]);

      // Add meeting_id to response for clarity
      if (is_array($response)) {
        $response['meeting_id'] = $meeting_id;
      }

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to add registrant to Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Removes a registrant from a meeting.
   *
   * @param string $meeting_id
   *   The meeting ID.
   * @param string $registrant_id
   *   The registrant ID.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function removeRegistrant(string $meeting_id, string $registrant_id, string $resource_type = self::RESOURCE_TYPE_MEETING): bool {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "webinars/{$meeting_id}/registrants/{$registrant_id}" : "meetings/{$meeting_id}/registrants/{$registrant_id}";

      $this->zoomClient->delete($endpoint);

      return TRUE;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to remove registrant from Zoom @resource: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return FALSE;
    }
  }

  /**
   * Lists registrants for a meeting.
   *
   * @param string $meeting_id
   *   The meeting ID.
   * @param string $status
   *   The registration status (approved, pending, denied).
   * @param int $page_size
   *   The page size.
   * @param string $next_page_token
   *   The next page token.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The registrants list or NULL on failure.
   */
  public function listRegistrants(string $meeting_id, string $status = self::REGISTRATION_STATUS_APPROVED, int $page_size = self::DEFAULT_PAGE_SIZE, string $next_page_token = '', string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "webinars/{$meeting_id}/registrants" : "meetings/{$meeting_id}/registrants";

      $query_params = [
        'status' => $status,
        'page_size' => $page_size,
      ];
      if (!empty($next_page_token)) {
        $query_params['next_page_token'] = $next_page_token;
      }

      return $this->zoomClient->get($endpoint, [
        'query' => $query_params,
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to list Zoom @resource registrants: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets a meeting report.
   *
   * @param string $meeting_id
   *   The meeting ID or UUID.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The meeting report or NULL on failure.
   */
  public function getMeetingReport(string $meeting_id, string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "report/webinars/{$meeting_id}" : "report/meetings/{$meeting_id}";

      return $this->zoomClient->get($endpoint);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Zoom @resource report: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets a participant report for a meeting.
   *
   * @param string $meeting_id
   *   The meeting ID or UUID.
   * @param int $page_size
   *   The page size.
   * @param string $next_page_token
   *   The next page token.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The participant report or NULL on failure.
   */
  public function getParticipantReport(string $meeting_id, int $page_size = self::DEFAULT_PAGE_SIZE, string $next_page_token = '', string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      $endpoint = ($resource_type === self::RESOURCE_TYPE_WEBINAR) ? "report/webinars/{$meeting_id}/participants" : "report/meetings/{$meeting_id}/participants";

      $query_params = [
        'page_size' => $page_size,
      ];
      if (!empty($next_page_token)) {
        $query_params['next_page_token'] = $next_page_token;
      }

      return $this->zoomClient->get($endpoint, [
        'query' => $query_params,
      ]);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Zoom @resource participant report: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets meeting recording details.
   *
   * @param string $meeting_id
   *   The meeting ID or UUID.
   * @param string $resource_type
   *   The resource type: 'meeting' or 'webinar'.
   *
   * @return array|null
   *   The recording details or NULL on failure.
   */
  public function getRecordingDetails(string $meeting_id, string $resource_type = self::RESOURCE_TYPE_MEETING): ?array {
    try {
      // Note: Recordings endpoint is the same for both meetings and webinars
      $response = $this->zoomClient->get("meetings/{$meeting_id}/recordings");

      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to get Zoom @resource recording details: @message', [
        '@resource' => $resource_type,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
