<?php

namespace Drupal\ecoindex\Plugin\Field\FieldWidget;

use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Url;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Plugin implementation of the 'ecoindex' widget.
 *
 * @FieldWidget(
 *   id = "ecoindex_widget",
 *   label = @Translation("EcoIndex widget"),
 *   field_types = {
 *     "ecoindex"
 *   }
 * )
 */
class EcoIndexWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The request stack.
   */
  protected RequestStack $requestStack;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a EcoIndexWidget object.
   *
   * @param string $plugin_id
   *   The plugin_id for the widget.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the widget is associated.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, RequestStack $request_stack, RouteMatchInterface $route_match) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->requestStack = $request_stack;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('request_stack'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    /** @var \Drupal\ecoindex\Plugin\Field\FieldType\EcoIndexItem $item */
    $item = $items[$delta];

    $element += [
      '#type' => 'details',
      '#title' => $this->t('EcoIndex'),
      '#open' => FALSE,
    ];

    $element['element_count'] = [
      '#title' => $this->t('Element Count'),
      '#type' => 'number',
      '#min' => 0,
      '#default_value' => $item->element_count ?? 0,
    ];

    $element['request_count'] = [
      '#title' => $this->t('Request Count'),
      '#type' => 'number',
      '#min' => 0,
      '#default_value' => $item->request_count ?? 0,
    ];

    $element['total_size_kb'] = [
      '#title' => $this->t('Total Size (KB)'),
      '#type' => 'number',
      '#step' => 0.01,
      '#min' => 0,
      '#default_value' => $item->total_size_kb ?? 0,
    ];

    $element['score'] = [
      '#title' => $this->t('Score'),
      '#type' => 'number',
      '#min' => 0,
      '#max' => 100,
      '#default_value' => $item->score ?? 0,
    ];

    $element['grade'] = [
      '#title' => $this->t('Grade'),
      '#type' => 'select',
      '#options' => [
        '' => '',
        'A' => 'A',
        'B' => 'B',
        'C' => 'C',
        'D' => 'D',
        'E' => 'E',
        'F' => 'F',
        'G' => 'G',
      ],
      '#default_value' => $items[$delta]->grade ?? '',
    ];

    $node = $this->routeMatch->getParameter('node');
    if ($node instanceof NodeInterface) {
      $url = Url::fromRoute('ecoindex.preview', ['node' => $node->id()]);
      $element['grade']['#description'] = '<p><a href="' . $url->toString() . '" target="_blank">Refresh EcoIndex score</a></p>';
    }

    return $element;
  }

}
