<?php

namespace Drupal\edit_plus\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\lb_plus\SectionStorageHandler;
use Symfony\Component\HttpFoundation\Request;
use Drupal\edit_plus\Event\BlockPropertiesEvent;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Layout Builder block form alter.
 */
class LayoutBuilderBlockFormAlter {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  protected Request $request;

  public function __construct(
    protected SectionStorageHandler $sectionStorageHandler,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Layout Builder block form alter.
   *
   * This form alter is intended to ease the use of adding properties to inline
   * blocks for styling. The properties can then be used for things like choosing
   * different templates or setting classes, etc.
   *
   * Please see https://www.drupal.org/project/plus_suite_extras for example usage.
   */
  public function formAlter(&$form, FormStateInterface $form_state, string $form_id) {
    if (!in_array($form_id, [
      'layout_builder_update_block',
      'layout_builder_add_block',
    ])) {
      return;
    }
    [
      $main_section_storage,
      $current_section_storage,
      $nested_storage_path,
      $section_component,
      $delta,
      $uuid,
      $inline_block,
      $inline_block_configuration,
      $block_content,
    ] = $this->getArgs($form_state);

    $block_properties_event = $this->eventDispatcher->dispatch(new BlockPropertiesEvent($main_section_storage, $current_section_storage, $section_component, $block_content, $inline_block_configuration, $form, $form_state), BlockPropertiesEvent::class);
    $form_state->set('block_properties', $block_properties_event);
    $form = $block_properties_event->getForm();
    $form_state = $block_properties_event->getFormState();

    $form['#submit'][] = [$this, 'formSubmit'];
  }

  public function formSubmit($form, FormStateInterface $form_state) {
    [
      $main_section_storage,
      $current_section_storage,
      $nested_storage_path,
      $section_component,
      $delta,
      $uuid,
      $inline_block,
      $inline_block_configuration,
      $block_content,
    ] = $this->getArgs($form_state);

    $block_properties_event = $form_state->get('block_properties');
    $block_properties = $block_properties_event->getBlockProperties();
    if (!empty($block_properties)) {
      foreach ($block_properties as $block_property) {
        $value = $form_state->getValue($block_property);
        $key = is_array($block_property) ? end($block_property) : $block_property;
        $inline_block_configuration[$key] = $value;
      }
      $section_component->setConfiguration($inline_block_configuration);
      $this->sectionStorageHandler->updateSectionStorage($main_section_storage, $nested_storage_path, $current_section_storage);
    }
  }

  private function getArgs(FormStateInterface $form_state) {
    [$main_section_storage, $delta, $region, $uuid, $nested_storage_path] = $form_state->getBuildInfo()['args'];
    $current_section_storage = $this->sectionStorageHandler->getCurrentSectionStorage($main_section_storage, $nested_storage_path);
    $section_component = $current_section_storage->getSection($delta)->getComponent($uuid);
    $inline_block = $section_component->getPlugin();
    $inline_block_configuration = $inline_block->getConfiguration();
    $block_content = $this->sectionStorageHandler->getBlockContent($inline_block);
    return [
      $main_section_storage,
      $current_section_storage,
      $nested_storage_path,
      $section_component,
      $delta,
      $uuid,
      $inline_block,
      $inline_block_configuration,
      $block_content,
    ];
  }

}
