<?php

namespace Drupal\edit_plus_header_block\EventSubscriber;

use Drupal\lb_plus\Event\PlaceBlockEvent;
use Drupal\edit_plus\Event\BlockPropertiesEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Heading.
 *
 * An example of how to alter the Inline Block form to be able to change classes
 * on the block for styling.
 */
class Heading implements EventSubscriberInterface {

  /**
   * On post place block form build.
   *
   * @param \Drupal\lb_plus\Event\PlaceBlockEvent $event
   *  Dispatched after the blocks content has been auto-generated.
   *
   * @return void
   */
  public function onPostPlaceBlockFormBuild(PlaceBlockEvent $event) {
    if ($event->getBlockPluginId() === 'inline_block' && $event->getBundle() === 'heading') {
      $block_plugin = $event->getBlockPlugin();
      $configuration = $block_plugin->getConfiguration();
      $configuration['label_display'] = 'visible';
      $configuration['label'] = $block_plugin->label();
      $configuration['label_tag'] = 'h2';
      $block_plugin->setConfiguration($configuration);
    }
  }

  /**
   * On block properties.
   *
   * @param \Drupal\edit_plus\Event\BlockPropertiesEvent $event
   *
   * @return void
   */
  public function onBlockProperties(BlockPropertiesEvent $event): void {
    $inline_block_configuration = $event->getInlineBlockConfiguration();
    if (empty($inline_block_configuration['id']) || $inline_block_configuration['id'] !== 'inline_block:heading') {
      return;
    }
    $form = $event->getForm();
    $config = $event->getInlineBlockConfiguration();

    $form['settings']['admin_label']['#access'] = FALSE;
    // Header blocks need the block's label, so we're going to remove the ability to
    // disable it from the form.
    $form['settings']['label']['#access'] = TRUE;
    $form['settings']['label_display']['#value'] = TRUE;
    $form['settings']['label_display']['#access'] = FALSE;
    $form['settings']['label_display']['#default_value'] = TRUE;

    $form['settings']['horizontal_alignment'] = [
      '#type' => 'select',
      '#title' => t('Alignment'),
      '#options' => [
        'text-left' => t('Left'),
        'text-center' => t('Center'),
        'text-right' => t('Right'),
      ],
      '#default_value' => !empty($config['horizontal_alignment']) ? $config['horizontal_alignment'] : 'left',
    ];

    $form['settings']['title_size'] = [
      '#type' => 'select',
      '#title' => t('Size'),
      '#options' => [
        'text-jumbo1' => t('Jumbo 65px'),
        'text-jumbo2' => t('Jumbo 54px'),
        'text-jumbo3' => t('Jumbo 45px'),
        'text-jumbo4' => t('Jumbo 38px'),
        'text-large1' => t('Large 32px'),
        'text-large2' => t('Large 28px'),
        'text-large3' => t('Large 24px'),
        'text-large4' => t('Large 21px'),
        'text-large5' => t('Large 18px'),
        'text-large6' => t('Large 16px'),
      ],
      '#default_value' => $config['title_size'] ?? NULL,
      '#empty_option' => t('Default'),
    ];

    $event->setForm($form);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      PlaceBlockEvent::class => ['onPostPlaceBlockFormBuild'],
      BlockPropertiesEvent::class => ['onBlockProperties'],
    ];
  }

}
