<?php

namespace Drupal\edit_plus\Event;

use Drupal\layout_builder\Plugin\SectionStorage\OverridesSectionStorage;
use Drupal\block_content\BlockContentInterface;
use Drupal\layout_builder\SectionComponent;
use Drupal\Component\EventDispatcher\Event;
use Drupal\Core\Form\FormStateInterface;

/**
 * Block properties event.
 *
 * A wrapper around the layout_builder_add_block and layout_builder_update_block
 * form alters. Allows modules to add custom block properties to inline blocks.
 * Add form items to $form['settings'] and they will show up in the Change tool
 * sidebar. Settings are automatically saved. If not, try adding the form item
 * path of where it is located in $form_state->getValue with addBlockProperty.
 */
class BlockPropertiesEvent extends Event {

  /**
   * An array of block properties (the form items that were added to this event)
   * that need to be saved on form submit.
   *
   * @var array
   *   An array of $form_state->getValue() parameters.
   * @see FormStateInterface::getValue()
   */
  protected array $blockProperties = [];

  public function __construct(
    private readonly array $inlineBlockConfiguration,
    private array $form,
    private FormStateInterface $formState,
  ) {}

  public function getForm(): array {
    return $this->form;
  }

  public function setForm(array $form): void {
    $this->form = $form;
  }

  /**
   * @return \Drupal\Core\Form\FormStateInterface
   */
  public function getFormState(): FormStateInterface {
    return $this->formState;
  }

  /**
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   */
  public function setFormState(FormStateInterface $form_state): void {
    $this->formState = $form_state;
  }

  public function getInlineBlockConfiguration(): array {
    return $this->inlineBlockConfiguration;
  }

  public function getBlockProperties(): array {
    return $this->blockProperties;
  }

  /**
   * Add block property.
   *
   * Informs LayoutBuilderBlockFormAlter::formSubmit that a form item was added
   * that needs to be saved.
   *
   * @param string|array $blockProperties
   *   Either a form item key or an array of key paths for $form_state->getValue.
   * @see FormStateInterface::getValue()
   */
  public function addBlockProperty($block_property): void {
    $this->blockProperties[] = $block_property;
  }

}
