<?php declare(strict_types = 1);

namespace Drupal\edit_plus\Controller;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Messenger\MessengerTrait;
use Drupal\Core\Controller\ControllerBase;
use Drupal\edit_plus\EditPlusMessagesTrait;
use Symfony\Component\HttpFoundation\Cookie;
use Symfony\Component\HttpFoundation\Request;
use Drupal\edit_plus\EditPlusTempstoreRepository;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Edit + tempstore controller.
 */
class Tempstore extends ControllerBase {

  use MessengerTrait;
  use EditPlusMessagesTrait;

  public function __construct(
    protected EditPlusTempstoreRepository $tempstoreRepository
  ) {}

  public static function create(ContainerInterface $container) {
    return new static (
      $container->get('edit_plus.tempstore_repository'),
    );
  }

  /**
   * Save tempstore entities.
   */
  public function save(Request $request) {
    $first_entity = $this->loopThroughEntities($request, function ($entity) {
      $tempstore_entity = $this->tempstoreRepository->get($entity);
      $tempstore_entity->save();
      $this->tempstoreRepository->delete($entity);
    });

    $destination = \Drupal::destination();
    $use_path = $request->query->get('edit_mode_use_path');
    if ($use_path) {
      // Use a specified path as the main entity is not what should be loaded.
      $destination->set($use_path);
    } elseif ($first_entity) {
      // If the page title was changed, the autogenerated URL could be different.
      $url = $first_entity->toUrl()->toString();
      $request->query->set('destination', $url);
      $destination->set($url);
    }
    $destination = $destination->get();
    $response = new RedirectResponse($destination);
    $response->headers->setCookie((new Cookie('navigationMode', 'none', 0, $destination, NULL, FALSE, FALSE)));
    return $response;
  }

  /**
   * Delete tempstore entities.
   */
  public function delete(Request $request) {
    $this->loopThroughEntities($request, function ($entity) {
      $this->tempstoreRepository->delete($entity);
    });

    $this->clearUnsavedChangesMessage();

    return new RedirectResponse(\Drupal::destination()->get());
  }

  /**
   * Loop through entities.
   *
   * Gets the relevant entities from the request and loops through them.
   *
   * @param $request
   *   The current request.
   * @param $callable
   *   The function to call.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The next from the entity parameter.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *@see delete() or save().
   *
   */
  /**
   * @param $request
   * @param $callable
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  private function loopThroughEntities($request, $callable): ?EntityInterface {
    $entities = explode('::', $request->get('entities'));
    if (empty($entities)) {
      throw new \InvalidArgumentException('No entities to save.');
    }
    $first_entity = NULL;
    foreach ($entities as $entity) {
      [$entity_type_id, $entity_id] = explode('.', $entity);
      $entity = $this->entityTypeManager()->getStorage($entity_type_id)->load($entity_id);
      Cache::invalidateTags([getCacheTag($entity)]);
      $callable($entity);
      if (!$first_entity) {
        $first_entity = $entity;
      }
    }
    return $first_entity;
  }

}
