(($, Drupal) => {

  const DefaultSidebar = Drupal.NavigationPlus.DefaultSidebar;
  const sidebarManager = Drupal.NavigationPlus.SidebarManager;

  /**
   * Edit Plus Sidebar Plugin
   *
   * Manages the Edit Plus form sidebars that appear when editing fields
   * with the Change tool. This sidebar contains form widgets for editing
   * field values inline.
   */
  class EditPlusSidebar extends DefaultSidebar {
    type = 'edit_plus_form';

    /**
     * Close the Edit Plus form sidebar.
     *
     * Overrides DefaultSidebar to add validation logic.
     * Can reject if there is an invalid form item that needs to be fixed
     * or discarded before closing.
     *
     * @param {string|null} id
     *   The specific sidebar instance ID to close
     *
     * @returns {Promise}
     *   Resolves when the sidebar is closed, rejects if there is an invalid
     *   form item.
     */
    close(id = null) {
      return new Promise((resolve, reject) => {
        if (Drupal.EditPlus.hasInvalidFormItem()) {
          Drupal.EditPlus.warnAboutInvalidFormItem();
          reject('Invalid form item - cannot close sidebar');
          return;
        }

        // Safe to close - blur the current element if any.
        if (Drupal.EditPlus.CurrentlyEditingElement) {
          Drupal.EditPlus.CurrentlyEditingElement.plugin.blur().then(() => {
            this.closeSidebar(id);
            resolve();
          }).catch((error) => {
            if (error !== 'HTML validation failed') {
              console.error(error);
            }
            reject(error);
          });
        } else {
          this.closeSidebar(id);
          resolve();
        }
      });
    }

    /**
     * Actually close the sidebar element.
     *
     * @param {string|null} id
     *   The specific sidebar instance ID to close
     */
    closeSidebar(id = null) {
      const sidebar = this.getElement(id);
      if (sidebar) {
        this.hideSidebarElement(sidebar);
      }
    }
  }

  sidebarManager.registerPlugin(new EditPlusSidebar());

})(jQuery, Drupal);
