<?php

namespace Drupal\edit_plus_lb\Entity;

use Drupal\Core\Render\Element;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Plugin\Context\EntityContext;
use Drupal\Core\Entity\Entity\EntityViewDisplay;
use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay as BaseEntityViewDisplay;

/**
 * Provides an entity view display entity that has a layout.
 */
class LayoutBuilderEntityViewDisplay extends BaseEntityViewDisplay {

  /**
   * {@inheritdoc}
   */
  protected function buildSections(FieldableEntityInterface $entity) {
    $build = parent::buildSections($entity);
    $build['#cache']['contexts'][] = 'user.permissions';
    if (!\Drupal::currentUser()->hasPermission('access inline editing')) {
      return $build;
    }

    $contexts = $this->getContextsForEntity($entity);
    $label = new TranslatableMarkup('@entity being viewed', [
      '@entity' => $entity->getEntityType()->getSingularLabel(),
    ]);
    $contexts['layout_builder.entity'] = EntityContext::fromEntity($entity, $label);

    $cacheability = new CacheableMetadata();
    $storage = $this->getSectionStorage()->findByContext($contexts, $cacheability);

    if ($storage) {
      foreach ($storage->getSections() as $delta => $section) {
        $this->getSectionAttributes($delta, $build[$delta], $storage);
      }
    }
    return $build;
  }

  protected function getSectionStorage() {
    return \Drupal::service('plugin.manager.layout_builder.section_storage');
  }

  /**
   * @param int|string $delta
   * @param array $build
   * @param $storage
   *
   * @return array
   */
  public static function getSectionAttributes(int|string $delta, array &$build, $storage): array {
    // Attribute the rendered sections so that the Edit + JS can make sense
    // of the page.
    $build['#attributes']['data-layout-builder-section-delta'] = $delta;
    $build['#attributes']['data-layout-builder-section-storage-type'] = $storage->getStorageType();
    $build['#attributes']['data-layout-builder-section-storage-id'] = $storage->getStorageId();
    return $build;
  }

  /**
   * Build Multiple.
   *
   * Fields configured in the "Edit Plus: Preserved Fields" section will not be
   * removed from the render array, allowing them to be edited inline with
   * the Change tool while Layout Builder manages the rest of the page structure.
   */
  public function buildMultiple(array $entities) {
    $fields_that_should_render = $this->getThirdPartySetting('edit_plus_lb', 'preserved_fields', []);

    // Call grandparent to build the initial render array with all fields.
    $build_list = EntityViewDisplay::buildMultiple($entities);

    // Override parent below.

    // Layout Builder can not be enabled for the '_custom' view mode that is
    // used for on-the-fly rendering of fields in isolation from the entity.
    if ($this->isCustomMode()) {
      return $build_list;
    }

    foreach ($entities as $id => $entity) {
      $build_list[$id]['_layout_builder'] = $this->buildSections($entity);

      // If there are any sections, remove all fields with configurable display
      // from the existing build EXCEPT those configured to be preserved.
      // Preserved fields can be edited inline using Edit Plus.
      if (!Element::isEmpty($build_list[$id]['_layout_builder'])) {
        foreach ($build_list[$id] as $name => $build_part) {
          $field_definition = $this->getFieldDefinition($name);
          if (
            $field_definition &&
            $field_definition->isDisplayConfigurable($this->displayContext) &&
            // Omit removal of configured fields.
            !in_array($name, $fields_that_should_render, TRUE)
          ) {
            unset($build_list[$id][$name]);
          }
        }
      }
    }

    return $build_list;
  }

}
