import * as effects from './edit_plus/effects.js'; // Don't remove!
import * as hotkeys from './edit_plus/edit-plus-hotkeys.js'; // Don't remove!
import * as entityForm from './edit_plus/entity-form.js';
import * as editableElement from './edit_plus/editable-element.js';
import * as pluginManager from './edit_plus/field-plugin-manager.js';

(($, Drupal, once, displace) => {

  Drupal.EditPlus = {};
  Drupal.EditPlus.EditMode = 'enabled';

  /**
   * Currently Editing Element.
   *
   * Remember the focused form item. Instead of calling replaceFormItemWithPageElement
   * onblur, lets track the focused form item so that clicking things in the
   * sidebar doesn't lose focus on the currently edited form item.
   */
  Drupal.EditPlus.CurrentlyEditingElement = null;

  /**
   * Editable Element clicked.
   *
   * @param fieldValue
   *   The editable element.
   */
  Drupal.EditPlus.EditableElementClicked = (fieldValue) => {
    if (fieldValue.hasAttribute('contenteditable')) {
      // We are editing a textfield.
      // @see textfield.js
      return false;
    }

    if (Drupal.EditPlus.hasInvalidFormItem()) {
      Drupal.EditPlus.warnAboutInvalidFormItem();
      return false;
    }

    // @todo Make the default of which form is loaded initially configurable and
    // @todo add a UI to switch between the referenced entities forms.
    // @todo Default to the first form, e.g. entity reference or media library
    // @todo not the referenced entity form or the media entity form.
    while (fieldValue.parentNode.closest('.edit-plus-field-value')) {
      fieldValue = fieldValue.parentNode.closest('.edit-plus-field-value');
    }

    const EditableElement = new editableElement.EditableElement(fieldValue.dataset.editPlusId);

    // Prevent clicks on a form item that is currently being updated.
    if (EditableElement.getPageElementHandle().classList.contains('disabled-updating')) {
      return false;
    }

    entityForm.getForm(EditableElement).then((response, status) => {
      displace();
      // Set the widget now that the form is loaded.
      EditableElement.setWidget();

      if (Drupal.EditPlus.CurrentlyEditingElement) {
        Drupal.EditPlus.CurrentlyEditingElement.plugin.blur(e).then((response, status) => {
          EditableElement.plugin.edit(EditableElement);
        }).catch(e => {
          if (e !== 'HTML validation failed') {
            console.error(e);
          }
        });
      } else {
        EditableElement.plugin.edit(EditableElement);
      }


    }).catch((error) => {
      console.error('An error occurred while trying to edit an element:', error);
    });
  }

  /**
   * Disable edit mode
   */
  Drupal.EditPlus.DisableEditMode = async () => {
    await Drupal.EditPlus.notUpdating();
    return new Promise((resolve, reject) => {
      if (Drupal.EditPlus.hasInvalidFormItem()) {
        Drupal.EditPlus.warnAboutInvalidFormItem();
        reject();
      } else {
        document.querySelectorAll('.edit-plus-form.navigation-plus-sidebar').forEach(sidebar => {
          sidebar.classList.add('navigation-plus-hidden');
          sidebar.removeAttribute('data-offset-right');
          displace();
        });

        resolve();
      }
    });
  }

  /**
   * Not updating.
   *
   * Wait till edit+ is done updating.
   *
   * @returns {Promise<unknown>}
   */
  Drupal.EditPlus.notUpdating = () => {
    return new Promise(resolve => {
      const intervalId = setInterval(() => {
        if (!entityForm.editPlusIsUpdating) {
          clearInterval(intervalId);
          resolve();
        }
      }, 100);
    });
  }

  /**
   * Has invalid form item.
   *
   * A field is currently being edited and needs correction before we can edit
   * another field.
   *
   * @returns {boolean}
   *   Whether there is an invalid form item that needs to be resolved.
   */
  Drupal.EditPlus.hasInvalidFormItem = () => {
    const isValid =  Drupal.EditPlus.CurrentlyEditingElement?.plugin.isValid();
    return typeof isValid === 'boolean' ? !isValid : false;
  }

  /**
   * Warn about invalid form item.
   *
   * Shows a warning message and scrolls to the currently editing element
   * when it has invalid form data. This prevents users from editing another
   * field, closing the sidebar, or exiting edit mode until they either fix
   * the validation error or press Escape to discard changes.
   *
   * If a server-side validation error was previously shown for this element,
   * the warning will include the original error message to remind the user
   * what needs to be fixed.
   *
   * The warning appears in three scenarios:
   * - Attempting to edit a different field
   * - Attempting to close the edit plus sidebar
   * - Attempting to disable edit mode
   *
   * @see Drupal.EditPlus.hasInvalidFormItem()
   */
  Drupal.EditPlus.warnAboutInvalidFormItem = () => {
    const element = Drupal.EditPlus.CurrentlyEditingElement;
    const elementId = element.info.elementId;

    // Look up the most recent error for this element from localStorage.
    const key = 'navigation_plus_notifications';
    let history = [];
    try {
      history = JSON.parse(localStorage.getItem(key) || '[]');
    } catch (e) {
      history = [];
    }

    // Find the most recent error or warning for this element.
    const relevantNotifications = history
      .filter(n => n.elementId === elementId && (n.type === 'error' || n.type === 'warning'))
      .reverse();

    let message = Drupal.t('Please fix the invalid form item or press Esc to discard changes.');
    if (relevantNotifications.length > 0) {
      const originalError = relevantNotifications[0].message;
      // Combine original error with instruction.
      message = `${message}
      ${originalError}`;
    }

    let target = document.querySelector('#' + Drupal.EditPlus.CurrentlyEditingElement.getSizedPlaceholderId());
    if (!target) {
      target = Drupal.EditPlus.CurrentlyEditingElement.getPageElementHandle();
    }
    const editMode = Drupal.NavigationPlus.ModeManager.getPlugin('edit');
    editMode.message(Drupal.t(message), 'warning', {
      duration: 3000,
      scroll: false,
    });
    editMode.scrollToElement(target);
  }

  // Register field plugins.
  pluginManager.default($, Drupal, once);

  /**
   * Disables interactive elements while in edit mode.
   */
  Drupal.EditPlus.DoNothing = (e) => {
    e.preventDefault();
    e.stopPropagation();
  };

  /**
   * An HTML5 validation message.
   *
   * @param {string} [message]
   *   (optional) The message shown on the UI.
   * @return {string}
   *   The HTML markup for the validation message.
   */
  Drupal.theme.editPlusValidationMessage = (message) => {
    return `<div class="edit-plus-validation-message">${message}</div>`;
  };

  // Silence the sizedPlaceholderHeightObserver benign console error in Safari. It's annoying.
  window.addEventListener('error', e => {
    if (e.message === 'ResizeObserver loop completed with undelivered notifications.') {
      e.stopPropagation();
      e.preventDefault();
    }
  });

})(jQuery, Drupal, once, Drupal.displace);




