<?php

declare(strict_types=1);

namespace Drupal\edit_plus\EventSubscriber;

use Drupal\edit_plus\Event\NoChangeTool;
use Drupal\navigation_plus\Event\EditableFieldAttributes;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Adds the required attributes needed to make the field inline editable.
 */
class RenderEditableFieldAttributes implements EventSubscriberInterface {

  public function __construct(
    private readonly EventDispatcherInterface  $eventDispatcher,
  ) {}

  public function onEditableFieldAttributes(EditableFieldAttributes $event) {

    $something_wants = $this->eventDispatcher->dispatch(new NoChangeTool($event), NoChangeTool::class);
    if ($something_wants->noChangeTool()) {
      return;
    }

    $variables =& $event->getVariables();
    $variables['#cache']['contexts'][] = 'user.permissions';
    if (!\Drupal::currentUser()->hasPermission('access inline editing')) {
      return;
    }

    $element = &$variables['element'];
    /** @var \Drupal\Core\Entity\ContentEntityInterface $entity */
    $entity = $element['#object'];

    $field_definition = $entity->getFieldDefinition($element['#field_name']);
    // Check if inline editing has been disabled for this field.
    if (method_exists($field_definition, 'getThirdPartySettings')) {
      $third_party_settings = $field_definition->getThirdPartySettings('edit_plus');
      if (!empty($third_party_settings['disable'])) {
        return;
      }
    }

    $view_mode = $element['#view_mode'];
    if ($view_mode === '_custom') {
      // Support layout builder field blocks.
      if (!empty($element['#third_party_settings']['layout_builder']['view_mode'])) {
        $view_mode = $element['#third_party_settings']['layout_builder']['view_mode'];
      }
      else {
        // Layout Builder (and Edit+) only supports view modes, not dynamically defined
        // "display options" (which \Drupal\Core\Field\FieldItemListInterface::view()
        // always names the "_custom" view mode).
        // @see \Drupal\Core\Field\FieldItemListInterface::view()
        // @see https://www.drupal.org/node/2120335
        return;
      }
    }
    // Get the actual view mode.
    $view_mode = _navigation_plus_get_view_mode($entity, $view_mode);
    $definition = $entity->getFieldDefinition($element['#field_name']);
    $main_property = method_exists($definition, 'getMainPropertyName') ? $definition->getMainPropertyName() : 'value';
    $id = navigation_plus_entity_identifier($entity);

    $variables['attributes']['data-edit-plus-field-value-wrapper'] = sprintf('%s::%s::%s::%s', $entity->getEntityTypeId(), $id, $element['#field_name'], $main_property);

    if (!empty($variables['items'])) {
      foreach ($variables['items'] as $delta => &$item) {
        $item['content']['edit_mode_wrapper']['#attributes']['class'] = ['edit-plus-field-value'];
        $item['content']['edit_mode_wrapper']['#attributes']['data-edit-plus-id'] = sprintf('%s::%s::%s::%s::%s::%s::%s::%s::%s', $entity->getEntityTypeId(), $entity->bundle(), $id, $element['#field_name'], $delta, $element['#language'], $element['#field_type'], $element['#formatter'], $main_property);
        $item['content']['edit_mode_wrapper']['#attributes']['data-edit-plus-page-element-id'] = sprintf('%s::%s::%s::%s::%s', $entity->getEntityTypeId(), $id, $element['#field_name'], $delta, $main_property);
        $item['content']['edit_mode_wrapper']['#cache']['tags'][] = getCacheTag($entity);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      EditableFieldAttributes::class => ['onEditableFieldAttributes', 200],
    ];
  }

}
