<?php

namespace Drupal\edit_plus_header_block\EventSubscriber;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Utility\NestedArray;
use Drupal\lb_plus\Event\PlaceBlockEvent;
use Drupal\edit_plus\Event\BlockPropertiesEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Heading.
 *
 * An example of how to alter the Inline Block form to be able to change classes
 * on the block for styling.
 */
class Heading implements EventSubscriberInterface {

  /**
   * On post place block form build.
   *
   * @param \Drupal\lb_plus\Event\PlaceBlockEvent $event
   *  Dispatched after the blocks content has been auto-generated.
   *
   * @return void
   */
  public function onPostPlaceBlockFormBuild(PlaceBlockEvent $event) {
    if ($event->getBlockPluginId() === 'inline_block' && $event->getBundle() === 'heading') {
      $block_plugin = $event->getBlockPlugin();
      $configuration = $block_plugin->getConfiguration();
      $configuration['label_display'] = 'visible';
      $configuration['label'] = $block_plugin->label();
      $configuration['label_tag'] = 'h2';
      $configuration['title_size'] = 'text-jumbo2';
      $block_plugin->setConfiguration($configuration);
    }
  }

  /**
   * On block properties.
   *
   * @param \Drupal\edit_plus\Event\BlockPropertiesEvent $event
   *
   * @return void
   */
  public function onBlockProperties(BlockPropertiesEvent $event): void {
    $inline_block_configuration = $event->getInlineBlockConfiguration();
    if (empty($inline_block_configuration['id']) || $inline_block_configuration['id'] !== 'inline_block:heading') {
      return;
    }
    $form = $event->getForm();
    $config = $event->getInlineBlockConfiguration();

    $form['settings']['admin_label']['#access'] = FALSE;
    // Header blocks need the block's label, so we're going to remove the ability to
    // disable it from the form.
    $form['settings']['label']['#access'] = TRUE;
    $form['settings']['label_display']['#value'] = TRUE;
    $form['settings']['label_display']['#access'] = FALSE;
    $form['settings']['label_display']['#default_value'] = TRUE;

    $form['settings']['horizontal_alignment'] = [
      '#type' => 'select',
      '#title' => t('Alignment'),
      '#options' => [
        'text-left' => t('Left'),
        'text-center' => t('Center'),
        'text-right' => t('Right'),
      ],
      '#default_value' => !empty($config['horizontal_alignment']) ? $config['horizontal_alignment'] : 'left',
    ];

    $form['settings']['title_size'] = [
      '#type' => 'select',
      '#title' => t('Size'),
      '#options' => [
        'text-jumbo1' => t('Jumbo 65px'),
        'text-jumbo2' => t('Jumbo 54px'),
        'text-jumbo3' => t('Jumbo 45px'),
        'text-jumbo4' => t('Jumbo 38px'),
        'text-large1' => t('Large 32px'),
        'text-large2' => t('Large 28px'),
        'text-large3' => t('Large 24px'),
        'text-large4' => t('Large 21px'),
        'text-large5' => t('Large 18px'),
        'text-large6' => t('Large 16px'),
      ],
      '#default_value' => $config['title_size'] ?? NULL,
      '#empty_option' => t('Default'),
    ];

    $form['settings']['link_enabled'] = [
      '#type' => 'checkbox',
      '#title' => t('Link'),
      '#default_value' => !empty($config['url']),
    ];

    $form['settings']['url'] = [
      '#type' => 'url',
      '#title' => t('URL'),
      '#default_value' => $config['url'] ?? '',
      '#element_validate' => [[static::class, 'validateUrlOrPath']],
      '#states' => [
        'required' => [
          ':input[name="settings[link_enabled]"]' => ['checked' => TRUE],
        ],
        'visible' => [
          ':input[name="settings[link_enabled]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['#submit'] = array_merge([[$this, 'formSubmit']], $form['#submit']);

    $event->setForm($form);
  }

  public function formSubmit($form, FormStateInterface $form_state) {
    // @see Drupal\edit_plus\Form\LayoutBuilderBlockFormAlter->formSubmit
    // for where the block properties are saved. Let's special case the link
    // checkbox by clearing the URL value.
    if ($form_state->getValue(['settings', 'link_enabled']) === 0) {
      $input = $form_state->getUserInput();
      $input['settings']['url'] = NULL;
      $form_state->setUserInput($input);
      $form_state->setValue(['settings', 'url'], NULL);
      $url_form_item = &NestedArray::getValue($form, ['settings', 'url']);
      $url_form_item['#default_value'] = NULL;
      $form_state->setTemporaryValue('updateForm', TRUE);
    }
  }

  /**
   * Custom validation for URL or relative path.
   */
  public static function validateUrlOrPath(&$element, FormStateInterface $form_state, &$complete_form) {
    $value = trim($element['#value']);
    $form_state->setValueForElement($element, $value);

    if ($value !== '') {
      // Check if it's a valid absolute URL
      if (UrlHelper::isValid($value, TRUE)) {
        return;
      }

      // Check if it's a valid relative path (must start with /)
      if ((
          str_starts_with($value, '/') ||
          str_starts_with($value, '#')
        ) && UrlHelper::isValid($value, FALSE)) {
        return;
      }

      $form_state->setError($element, t('The URL %url is not valid. Enter an absolute URL (http://example.com) or relative path starting with "/" (/page).', ['%url' => $value]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      PlaceBlockEvent::class => ['onPostPlaceBlockFormBuild'],
      BlockPropertiesEvent::class => ['onBlockProperties'],
    ];
  }

}
