<?php

namespace Drupal\edit_plus_lb\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;

/**
 * Alters Entity View Display Form.
 *
 * Adds form options to prevent configured fields from being removed from the
 * render array on pages managed by Layout Builder.
 */
class EntityViewDisplayForm implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new EntityViewDisplayForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entityFieldManager
   *   The entity field manager.
   */
  public function __construct(
    private EntityTypeManagerInterface $entityTypeManager,
    private EntityFieldManagerInterface $entityFieldManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('entity_field.manager')
    );
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  public function formAlter(&$form, FormStateInterface $form_state) {
    /** @var \Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay $entity */
    $entity = $form_state->getFormObject()->getEntity();

    if (!$entity->isLayoutBuilderEnabled()) {
      return;
    }

    $entity_type_id = $entity->getTargetEntityTypeId();
    $bundle = $entity->getTargetBundle();

    $field_definitions = $this->entityFieldManager->getFieldDefinitions($entity_type_id, $bundle);

    $field_options = [];
    foreach ($field_definitions as $field_name => $field_definition) {
      // Include if field is display-configurable OR if it's already configured
      // in this entity view display.
      if ($field_definition->isDisplayConfigurable('view') || $entity->getComponent($field_name)) {
        $field_options[$field_name] = $field_definition->getLabel();
      }
    }

    $preserved_fields = $entity->getThirdPartySetting('edit_plus_lb', 'preserved_fields', []);

    $form['edit_plus_lb'] = [
      '#type' => 'details',
      '#title' => $this->t('Preserved Fields'),
      '#description' => $this->t('By default, Layout Builder removes all entity fields from the render array since they are typically managed as field blocks within the layout. Select fields below to preserve them in the render array outside of Layout Builder sections. Preserved fields can be edited with the Change tool and will appear in your theme\'s content variable, allowing you to position them independently from the Layout Builder UI for greater design flexibility.'),
      '#open' => !empty($preserved_fields),
      '#tree' => TRUE,
      '#weight' => 99,
    ];

    $form['edit_plus_lb']['preserved_fields'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Fields to preserve in render array'),
      '#options' => $field_options,
      '#default_value' => $preserved_fields,
    ];

    $form['#entity_builders'][] = [static::class, 'entityFormEntityBuild'];
  }

  /**
   * Entity builder for edit_plus_lb preserved fields settings.
   */
  public static function entityFormEntityBuild($entity_type, $entity, &$form, FormStateInterface $form_state) {
    /** @var \Drupal\layout_builder\Entity\LayoutBuilderEntityViewDisplay $entity */
    $preserved_fields = $form_state->getValue(['edit_plus_lb', 'preserved_fields']);
    $preserved_fields = array_filter($preserved_fields);

    if (!empty($preserved_fields)) {
      $entity->setThirdPartySetting('edit_plus_lb', 'preserved_fields', array_values($preserved_fields));
    }
    else {
      $entity->unsetThirdPartySetting('edit_plus_lb', 'preserved_fields');
    }
  }

}
