import * as entityForm from '../entity-form.js';
import * as fieldPluginBase from './field-plugin-base.js';

export default function ($ = jQuery, Drupal, once, dropZones) {

  /**
   * Media field plugin.
   */
  class MediaPlugin extends fieldPluginBase.FieldPluginBase {
    constructor() {
      super();
      this.originalValue = null;
    }
    supportedWidgets = [
      'media_library_widget',
    ];
    focusElement(EditableElement) {
      this.originalValue = this.getMediaWidgetValue(EditableElement);
      window.setFileDrag(false);
    }
    blurElement(EditableElement) {
      const newValue = this.getMediaWidgetValue(EditableElement);
      const isEmpty = this.isMediaValueEmpty(newValue);

      this.replaceFormItemWithPageElement(EditableElement);
      if (this.originalValue !== newValue) {
        entityForm.updateTempstore(EditableElement, isEmpty);
      }
      window.setFileDrag(true);
      return Promise.resolve();
    }

    getFocusedSelectors() {
      let selectors = super.getFocusedSelectors();
      selectors.push('.field--widget-media-library-widget', '#drupal-modal', '.ui-dialog-buttonpane');
      return selectors;
    }

    validationMessageHandle() {
      const EditableElement = Drupal.EditPlus.CurrentlyEditingElement;
      const formItem = EditableElement.getFormItem();
      return formItem.querySelector('.fieldset-wrapper') ?? formItem.querySelector('.fieldset__wrapper');
    }

    isValid(e = null) {
      const EditableElement = Drupal.EditPlus.CurrentlyEditingElement;
      if (!EditableElement.getFormItem().hasAttribute('required')) {
        return true;
      }
      const newValue = this.getMediaWidgetValue(EditableElement);
      const isValid = !this.isMediaValueEmpty(newValue);
      if (!isValid) {
        this.displayValidationMessage(Drupal.t('A media item is required.'));
      }
      return isValid;
    }

    getMediaWidgetValue(EditableElement) {
      return EditableElement.getFormItem()?.dataset?.editPlusMediaValue ?? null;
    }

    /**
     * Check if media value is empty.
     *
     * @param {string|null} mediaValue
     *   JSON-encoded array of media target IDs.
     *
     * @returns {boolean}
     *   True if the media value is empty (null, invalid JSON, or empty array).
     */
    isMediaValueEmpty(mediaValue) {
      if (mediaValue === null) {
        return true;
      }
      try {
        const targetIds = JSON.parse(mediaValue);
        return !Array.isArray(targetIds) || targetIds.length === 0;
      } catch (e) {
        // Invalid JSON, treat as empty
        return true;
      }
    }

    clearValidationMessageOnUserInput(validationMessageHandle) {
      // Do nothing. The validation message is removed when the media library
      // is used.
    }

  }

  /**
   * Register the default field plugin.
   */
  window.addEventListener('EditPlusFieldManager.RegisterPlugins', e => {
    e.detail.manager.registerPlugin(new MediaPlugin());
  });

}

