<?php

namespace Drupal\edit_plus\Form;

use Drupal\Core\Form\FormStateInterface;
use Drupal\layout_builder\Plugin\Block\InlineBlock;
use Drupal\lb_plus\SectionStorageHandler;
use Symfony\Component\HttpFoundation\Request;
use Drupal\edit_plus\Event\BlockPropertiesEvent;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Layout Builder block form alter.
 */
class LayoutBuilderBlockFormAlter {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  protected Request $request;

  public function __construct(
    protected SectionStorageHandler $sectionStorageHandler,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Layout Builder block form alter.
   *
   * This form alter is intended to ease the use of adding properties to inline
   * blocks for styling. The properties can then be used for things like choosing
   * different templates or setting classes, etc.
   */
  public function formAlter(&$form, FormStateInterface $form_state, string $form_id) {
    if (!in_array($form_id, [
      'layout_builder_update_block',
      'layout_builder_add_block',
    ])) {
      return;
    }
    $form['actions']['submit']['#submit'] = array_merge(
      [[$this, 'formSubmit']],
      $form['actions']['submit']['#submit'] ?? [],
      $form['#submit'] ?? []
    );

    [$main_section_storage, $delta, $region, $uuid, $nested_storage_path] = $form_state->getBuildInfo()['args'];
    $current_section_storage = $this->sectionStorageHandler->getCurrentSectionStorage($main_section_storage, $nested_storage_path);
    $section_component = $current_section_storage->getSection($delta)->getComponent($uuid);
    $inline_block = $section_component->getPlugin();
    if (!$inline_block instanceof InlineBlock) {
      return;
    }
    $inline_block_configuration = $inline_block->getConfiguration();
    $block_content = $this->sectionStorageHandler->getBlockContent($inline_block);
    $block_properties_event = $this->eventDispatcher->dispatch(new BlockPropertiesEvent($main_section_storage, $current_section_storage, $section_component, $block_content, $inline_block_configuration, $form, $form_state), BlockPropertiesEvent::class);

    $form_state = $block_properties_event->getFormState();

    // Let's determine what form items where added.
    $initial_settings = array_keys($form['settings']);
    $form = $block_properties_event->getForm();
    $settings_added = array_diff_key(array_keys($form['settings']), $initial_settings);
    $block_properties = [];
    foreach ($settings_added as $setting) {
      $block_properties[] = ['settings', $setting];
    }
    $block_properties = array_merge($block_properties, $block_properties_event->getBlockProperties());
    $form_state->set('block_properties', $block_properties);
  }

  public function formSubmit($form, FormStateInterface $form_state) {
    $form_object = $form_state->getFormObject();
    $p = new \ReflectionProperty($form_object, 'block');
    $p->setAccessible(TRUE);
    /** @var \Drupal\Component\Plugin\ConfigurableInterface $component */
    $inlineBlock = $p->getValue($form_object);

    $config = $inlineBlock->getConfiguration();
    $block_properties = $form_state->get('block_properties');
    if (!empty($block_properties)) {
      foreach ($block_properties as $block_property) {
        $value = $form_state->getValue($block_property);
        $key = is_array($block_property) ? end($block_property) : $block_property;
        $config[$key] = $value;
      }
      $inlineBlock->setConfiguration($config);
    }
  }

}
