/**
 * @file
 * Set Table Attributes Command.
 *
 * Command for setting table element attributes (id, dir, class).
 *
 * @module tableAdvanced/commands/SetTableAttributesCommand
 */

import { Command } from "ckeditor5/src/core";

/**
 * Command to set table attributes.
 *
 * Allows setting id, dir, and class attributes on table elements.
 */
export default class SetTableAttributesCommand extends Command {
  /**
   * @inheritdoc
   */
  refresh() {
    const { model } = this.editor;
    const { selection } = model.document;

    // Get table from current selection.
    const table = SetTableAttributesCommand._getTableFromSelection(selection);

    this.isEnabled = !!table;

    if (table) {
      // Ensure all values default to empty string if not set
      this.value = {
        id: table.getAttribute("tableId") || "",
        dir: table.getAttribute("tableDir") || "",
        class: table.getAttribute("tableClass") || ""
      };
    } else {
      // Default values when no table is selected
      this.value = {
        id: "",
        dir: "",
        class: ""
      };
    }
  }

  /**
   * @inheritdoc
   */
  execute(attributes = {}) {
    const { model } = this.editor;
    const { selection } = model.document;
    const table = SetTableAttributesCommand._getTableFromSelection(selection);

    if (!table) {
      return;
    }

    model.change(writer => {
      // Set or remove attributes.
      // Only set if value is a non-empty string.
      if (attributes.id && attributes.id.trim() !== "") {
        writer.setAttribute("tableId", attributes.id.trim(), table);
      } else {
        writer.removeAttribute("tableId", table);
      }

      if (
        attributes.dir &&
        attributes.dir.trim() !== "" &&
        attributes.dir !== "Not set"
      ) {
        writer.setAttribute("tableDir", attributes.dir.trim(), table);
      } else {
        writer.removeAttribute("tableDir", table);
      }

      if (attributes.class && attributes.class.trim() !== "") {
        writer.setAttribute("tableClass", attributes.class.trim(), table);
      } else {
        writer.removeAttribute("tableClass", table);
      }
    });
  }

  /**
   * Gets table element from selection.
   *
   * @param {module:engine/model/selection~Selection} selection
   *   The current selection.
   * @return {module:engine/model/element~Element|null}
   *   The table element or null.
   * @private
   */
  static _getTableFromSelection(selection) {
    const position = selection.getFirstPosition();

    if (!position) {
      return null;
    }

    // Find table in parent chain.
    return position.findAncestor("table");
  }
}
