<?php

declare(strict_types=1);

namespace Drupal\Tests\editor_advanced_table\Unit\Plugin\CKEditor5Plugin;

use Drupal\editor_advanced_table\Plugin\CKEditor5Plugin\TableAdvanced;
use Drupal\editor\EditorInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the TableAdvanced plugin.
 *
 * @coversDefaultClass \Drupal\editor_advanced_table\Plugin\CKEditor5Plugin\TableAdvanced
 * @group editor_advanced_table
 */
class TableAdvancedTest extends UnitTestCase {

  /**
   * The plugin under test.
   *
   * @var \Drupal\editor_advanced_table\Plugin\CKEditor5Plugin\TableAdvanced
   */
  protected TableAdvanced $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $configuration = [];
    $plugin_id = 'editor_advanced_table_table_advanced';
    $plugin_definition = [];

    $this->plugin = new TableAdvanced(
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * Tests default configuration.
   *
   * @covers ::defaultConfiguration
   */
  public function testDefaultConfiguration(): void {
    $config = $this->plugin->defaultConfiguration();

    $this->assertIsArray($config);
    $this->assertArrayHasKey('allow_classes', $config);
    $this->assertArrayHasKey('allow_id', $config);
    $this->assertArrayHasKey('allow_language_direction', $config);

    $this->assertTrue($config['allow_classes']);
    $this->assertTrue($config['allow_id']);
    $this->assertTrue($config['allow_language_direction']);
  }

  /**
   * Tests getDynamicPluginConfig with default configuration.
   *
   * @covers ::getDynamicPluginConfig
   */
  public function testGetDynamicPluginConfigDefaults(): void {
    $editor = $this->createMock(EditorInterface::class);

    $config = $this->plugin->getDynamicPluginConfig([], $editor);

    $this->assertArrayHasKey('tableAdvanced', $config);
    $this->assertTrue($config['tableAdvanced']['allowClasses']);
    $this->assertTrue($config['tableAdvanced']['allowId']);
    $this->assertTrue($config['tableAdvanced']['allowLanguageDirection']);
  }

  /**
   * Tests getDynamicPluginConfig with custom configuration.
   *
   * @covers ::getDynamicPluginConfig
   */
  public function testGetDynamicPluginConfigCustom(): void {
    $editor = $this->createMock(EditorInterface::class);

    // Set configuration.
    $reflection = new \ReflectionClass($this->plugin);
    $property = $reflection->getProperty('configuration');
    $property->setAccessible(TRUE);
    $property->setValue($this->plugin, [
      'allow_classes' => FALSE,
      'allow_id' => TRUE,
      'allow_language_direction' => FALSE,
    ]);

    $config = $this->plugin->getDynamicPluginConfig([], $editor);

    $this->assertArrayHasKey('tableAdvanced', $config);
    $this->assertFalse($config['tableAdvanced']['allowClasses']);
    $this->assertTrue($config['tableAdvanced']['allowId']);
    $this->assertFalse($config['tableAdvanced']['allowLanguageDirection']);
  }

  /**
   * Tests submitConfigurationForm sets configuration correctly.
   *
   * @covers ::submitConfigurationForm
   */
  public function testSubmitConfigurationFormSetsConfig(): void {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);

    // Create a value map for sequential getValue calls.
    $form_state->expects($this->exactly(3))
      ->method('getValue')
      ->willReturnOnConsecutiveCalls(
        TRUE,   // allow_classes
        FALSE,  // allow_id
        TRUE    // allow_language_direction
      );

    $this->plugin->submitConfigurationForm($form, $form_state);

    // Use reflection to check configuration was set correctly.
    $reflection = new \ReflectionClass($this->plugin);
    $property = $reflection->getProperty('configuration');
    $property->setAccessible(TRUE);
    $config = $property->getValue($this->plugin);

    $this->assertIsArray($config);
    $this->assertArrayHasKey('allow_classes', $config);
    $this->assertArrayHasKey('allow_id', $config);
    $this->assertArrayHasKey('allow_language_direction', $config);
  }

  /**
   * Tests validateConfigurationForm does not throw errors.
   *
   * @covers ::validateConfigurationForm
   */
  public function testValidateConfigurationForm(): void {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);

    // Should not throw any exceptions.
    $this->plugin->validateConfigurationForm($form, $form_state);
    $this->assertTrue(TRUE);
  }

  /**
   * Tests getDynamicPluginConfig with all features disabled.
   *
   * @covers ::getDynamicPluginConfig
   */
  public function testGetDynamicPluginConfigAllDisabled(): void {
    $editor = $this->createMock(EditorInterface::class);

    // Set all configuration to FALSE.
    $reflection = new \ReflectionClass($this->plugin);
    $property = $reflection->getProperty('configuration');
    $property->setAccessible(TRUE);
    $property->setValue($this->plugin, [
      'allow_classes' => FALSE,
      'allow_id' => FALSE,
      'allow_language_direction' => FALSE,
    ]);

    $config = $this->plugin->getDynamicPluginConfig([], $editor);

    $this->assertArrayHasKey('tableAdvanced', $config);
    $this->assertFalse($config['tableAdvanced']['allowClasses']);
    $this->assertFalse($config['tableAdvanced']['allowId']);
    $this->assertFalse($config['tableAdvanced']['allowLanguageDirection']);
  }

  /**
   * Tests getDynamicPluginConfig passes through static config.
   *
   * @covers ::getDynamicPluginConfig
   */
  public function testGetDynamicPluginConfigPassesStaticConfig(): void {
    $editor = $this->createMock(EditorInterface::class);

    $static_config = [
      'table' => [
        'contentToolbar' => ['tableColumn', 'tableRow'],
      ],
    ];

    $config = $this->plugin->getDynamicPluginConfig($static_config, $editor);

    $this->assertArrayHasKey('table', $config);
    $this->assertEquals(['tableColumn', 'tableRow'], $config['table']['contentToolbar']);
  }

}
