/**
 * @file
 * Table Advanced Editing plugin.
 *
 * Handles model-view conversion and commands for table attributes.
 *
 * @module tableAdvanced/TableAdvancedEditing
 */

import { Plugin } from "ckeditor5/src/core";
import SetTableAttributesCommand from "./commands/settableattributescommand";

/**
 * The Table Advanced editing feature.
 *
 * Introduces table attribute model elements and their converters.
 */
export default class TableAdvancedEditing extends Plugin {
  /**
   * @inheritdoc
   */
  static get pluginName() {
    return "TableAdvancedEditing";
  }

  /**
   * @inheritdoc
   */
  init() {
    const { editor } = this;
    const { schema } = editor.model;
    const { conversion } = editor;

    // Allow attributes on table element.
    schema.extend("table", {
      allowAttributes: ["tableId", "tableDir", "tableClass"]
    });

    // Upcast converters (view -> model).
    conversion.for("upcast").attributeToAttribute({
      view: {
        name: "table",
        key: "id"
      },
      model: "tableId"
    });

    conversion.for("upcast").attributeToAttribute({
      view: {
        name: "table",
        key: "dir"
      },
      model: "tableDir"
    });

    conversion.for("upcast").attributeToAttribute({
      view: {
        name: "table",
        key: "class"
      },
      model: "tableClass"
    });

    // Downcast converters (model -> view).
    // Only render attributes when they have non-empty values.
    conversion.for("downcast").add(dispatcher => {
      dispatcher.on("attribute:tableId:table", (evt, data, conversionApi) => {
        const { writer, mapper } = conversionApi;
        const viewElement = mapper.toViewElement(data.item);

        if (data.attributeNewValue) {
          writer.setAttribute("id", data.attributeNewValue, viewElement);
        } else {
          writer.removeAttribute("id", viewElement);
        }
      });

      dispatcher.on("attribute:tableDir:table", (evt, data, conversionApi) => {
        const { writer, mapper } = conversionApi;
        const viewElement = mapper.toViewElement(data.item);

        if (data.attributeNewValue) {
          writer.setAttribute("dir", data.attributeNewValue, viewElement);
        } else {
          writer.removeAttribute("dir", viewElement);
        }
      });
    });

    // Special handling for class attribute - need to find actual table element
    // in editing view when it's wrapped in a figure widget
    conversion.for("editingDowncast").add(dispatcher => {
      dispatcher.on(
        "attribute:tableClass:table",
        (evt, data, conversionApi) => {
          const { writer, mapper } = conversionApi;
          let viewElement = mapper.toViewElement(data.item);

          // When table properties plugin is enabled, tables are wrapped in figure
          // We need to find the actual table element inside the figure
          if (viewElement && viewElement.is("element", "figure")) {
            viewElement =
              Array.from(viewElement.getChildren()).find(child =>
                child.is("element", "table")
              ) || viewElement;
          }

          if (viewElement) {
            if (data.attributeNewValue) {
              writer.setAttribute("class", data.attributeNewValue, viewElement);
            } else {
              writer.removeAttribute("class", viewElement);
            }
          }
        },
        { priority: "low" }
      );
    });

    // Data downcast for class - simple attribute mapping (no figure wrapper in data view)
    conversion.for("dataDowncast").attributeToAttribute({
      model: "tableClass",
      view: "class"
    });

    // Register commands.
    this._defineCommands();
  }

  /**
   * Defines plugin commands.
   *
   * @private
   */
  _defineCommands() {
    const { editor } = this;
    const { commands } = editor;

    // Command to set table attributes.
    commands.add("setTableAttributes", new SetTableAttributesCommand(editor));
  }
}
