/**
 * @file
 * Table Advanced form view.
 *
 * Creates the form interface for editing table attributes.
 *
 * @module tableAdvanced/ui/TableAdvancedFormView
 */

import { View, ButtonView, submitHandler } from "ckeditor5/src/ui";

// Simple SVG icons for Save and Cancel buttons
const checkIcon =
  '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="M6.5 13.5L3.5 10.5 2 12l4.5 4.5L18 5l-1.5-1.5z"/></svg>';
const cancelIcon =
  '<svg viewBox="0 0 20 20" xmlns="http://www.w3.org/2000/svg"><path d="M11.591 10l4.72-4.72-1.591-1.59L10 8.409l-4.72-4.72-1.59 1.591L8.409 10l-4.72 4.72 1.591 1.59L10 11.591l4.72 4.72 1.59-1.591z"/></svg>';

/**
 * The form view for table advanced attributes.
 */
export default class TableAdvancedFormView extends View {
  /**
   * Constructs a TableAdvancedFormView.
   *
   * @param {module:utils/locale~Locale} locale
   *   The locale instance.
   * @param {Object} config
   *   The plugin configuration (optional, for future extensibility).
   */
  constructor(locale, config = {}) {
    super(locale);

    const { t } = locale;

    this.config = config;

    // Create header.
    this.headerView = this._createHeader();

    // Create fields based on configuration.
    if (config.allowId) {
      this.idInput = this._createIdInput();
    }

    if (config.allowLanguageDirection) {
      this.dirSelect = this._createDirSelect();
    }

    if (config.allowClasses) {
      this.classInput = this._createClassInput();
    }

    // Save button.
    this.saveButtonView = this._createButton(
      t("Save"),
      checkIcon,
      "ck-button-save"
    );
    this.saveButtonView.type = "submit";
    this.saveButtonView.withText = true;

    // Cancel button.
    this.cancelButtonView = this._createButton(
      t("Cancel"),
      cancelIcon,
      "ck-button-cancel"
    );
    this.cancelButtonView.withText = true;

    // Handle cancel.
    this.cancelButtonView.on("execute", () => {
      this.fire("cancel");
    });

    // Setup template.
    this.setTemplate({
      tag: "form",
      attributes: {
        class: ["ck", "ck-form", "ck-table-form", "ck-table-advanced-form"],
        tabindex: "-1"
      },
      children: this._getFormChildren()
    });
  }

  /**
   * Gets form children based on enabled features.
   *
   * @return {Array}
   *   Array of form elements.
   * @private
   */
  _getFormChildren() {
    const children = [this.headerView];

    if (this.idInput) {
      children.push(this.idInput);
    }

    if (this.dirSelect) {
      children.push(this.dirSelect);
    }

    if (this.classInput) {
      children.push(this.classInput);
    }

    // Create button row container
    const buttonRow = new View(this.locale);
    buttonRow.setTemplate({
      tag: "div",
      attributes: {
        class: ["ck", "ck-form__row", "ck-table-form__action-row"]
      },
      children: [this.saveButtonView, this.cancelButtonView]
    });

    children.push(buttonRow);

    return children;
  }

  /**
   * @inheritdoc
   */
  render() {
    super.render();

    submitHandler({
      view: this
    });
  }

  /**
   * Creates the form header.
   *
   * @return {module:ui/view~View}
   *   The header view.
   * @private
   */
  _createHeader() {
    const { t } = this.locale;
    const header = new View(this.locale);

    header.setTemplate({
      tag: "div",
      attributes: {
        class: ["ck", "ck-form__header"]
      },
      children: [
        {
          tag: "h2",
          attributes: {
            class: ["ck", "ck-form__header__label"]
          },
          children: [{ text: t("Advanced table properties") }]
        }
      ]
    });

    return header;
  }

  /**
   * Creates ID input field.
   *
   * @return {module:ui/view~View}
   *   The field view.
   * @private
   */
  _createIdInput() {
    const { t } = this.locale;
    const fieldView = new View(this.locale);

    fieldView.setTemplate({
      tag: "div",
      attributes: {
        class: ["ck", "ck-form__row", "ck-table-form__border-row"]
      },
      children: [
        {
          tag: "label",
          attributes: {
            class: ["ck", "ck-label"]
          },
          children: [{ text: t("Id") }]
        },
        {
          tag: "input",
          attributes: {
            type: "text",
            class: ["ck", "ck-input", "ck-input-text"],
            placeholder: t("e.g., pricing-table")
          }
        }
      ]
    });

    return fieldView;
  }

  /**
   * Creates language direction select dropdown.
   *
   * @return {module:ui/view~View}
   *   The field view.
   * @private
   */
  _createDirSelect() {
    const { t } = this.locale;
    const fieldView = new View(this.locale);

    fieldView.setTemplate({
      tag: "div",
      attributes: {
        class: ["ck", "ck-form__row", "ck-table-form__border-row"]
      },
      children: [
        {
          tag: "label",
          attributes: {
            class: ["ck", "ck-label"]
          },
          children: [{ text: t("Language Direction") }]
        },
        {
          tag: "select",
          attributes: {
            class: ["ck", "ck-input", "ck-input-select"]
          },
          children: [
            {
              tag: "option",
              attributes: { value: "", selected: "selected" },
              children: [{ text: t("Not set") }]
            },
            {
              tag: "option",
              attributes: { value: "ltr" },
              children: [{ text: t("Left to Right (LTR)") }]
            },
            {
              tag: "option",
              attributes: { value: "rtl" },
              children: [{ text: t("Right to Left (RTL)") }]
            }
          ]
        }
      ]
    });

    return fieldView;
  }

  /**
   * Creates CSS class input (free-form text).
   *
   * @return {module:ui/view~View}
   *   The field view.
   * @private
   */
  _createClassInput() {
    const { t } = this.locale;
    const fieldView = new View(this.locale);

    fieldView.setTemplate({
      tag: "div",
      attributes: {
        class: ["ck", "ck-form__row", "ck-table-form__border-row"]
      },
      children: [
        {
          tag: "label",
          attributes: {
            class: ["ck", "ck-label"]
          },
          children: [{ text: t("Stylesheet Classes") }]
        },
        {
          tag: "input",
          attributes: {
            type: "text",
            class: ["ck", "ck-input", "ck-input-text"],
            placeholder: t("e.g., table-striped table-bordered")
          }
        }
      ]
    });

    return fieldView;
  }

  /**
   * Creates a button.
   *
   * @param {string} label
   *   The button label.
   * @param {string} icon
   *   The button icon.
   * @param {string} className
   *   The CSS class name.
   * @return {module:ui/button/buttonview~ButtonView}
   *   The button instance.
   * @private
   */
  _createButton(label, icon, className) {
    const button = new ButtonView(this.locale);

    button.set({
      label,
      icon,
      tooltip: true
    });

    button.extendTemplate({
      attributes: {
        class: className
      }
    });

    return button;
  }

  /**
   * Gets current form values.
   *
   * @return {Object}
   *   Object with id, dir, and class values.
   */
  getValues() {
    const values = {};

    if (this.idInput && this.idInput.element) {
      const input = this.idInput.element.querySelector("input");
      if (input) {
        values.id = input.value.trim();
      }
    }

    if (this.dirSelect && this.dirSelect.element) {
      const select = this.dirSelect.element.querySelector("select");
      if (select) {
        // Ensure empty value or "Not set" text both result in empty string.
        const dirValue = select.value || "";
        values.dir = dirValue === "" || dirValue === "Not set" ? "" : dirValue;
      }
    }

    if (this.classInput && this.classInput.element) {
      const input = this.classInput.element.querySelector("input");
      if (input) {
        values.class = input.value.trim();
      }
    }

    return values;
  }

  /**
   * Sets form values.
   *
   * @param {Object} values
   *   Object with id, dir, and class values.
   */
  setValues(values) {
    if (this.idInput && this.idInput.element && values.id !== undefined) {
      const input = this.idInput.element.querySelector("input");
      if (input) {
        input.value = values.id;
      }
    }

    if (this.dirSelect && this.dirSelect.element) {
      const select = this.dirSelect.element.querySelector("select");
      if (select) {
        // Default to empty string (Not set) if dir is undefined, null, or empty
        const dirValue = values.dir !== undefined ? values.dir : "";
        select.value = dirValue || "";

        // Ensure the option is actually selected (handle edge cases)
        if (!select.value && dirValue === "") {
          select.selectedIndex = 0;
        }
      }
    }

    if (
      this.classInput &&
      this.classInput.element &&
      values.class !== undefined
    ) {
      const input = this.classInput.element.querySelector("input");
      if (input) {
        input.value = values.class;
      }
    }
  }

  /**
   * Focuses the first available input.
   */
  focus() {
    if (this.idInput && this.idInput.element) {
      const input = this.idInput.element.querySelector("input");
      if (input) {
        input.focus();
      }
    } else if (this.dirSelect && this.dirSelect.element) {
      const select = this.dirSelect.element.querySelector("select");
      if (select) {
        select.focus();
      }
    } else if (this.classInput && this.classInput.element) {
      const input = this.classInput.element.querySelector("input");
      if (input) {
        input.focus();
      }
    }
  }

  /**
   * Resets the form.
   */
  reset() {
    if (this.idInput && this.idInput.element) {
      const input = this.idInput.element.querySelector("input");
      if (input) {
        input.value = "";
      }
    }

    if (this.dirSelect && this.dirSelect.element) {
      const select = this.dirSelect.element.querySelector("select");
      if (select) {
        select.value = "";
      }
    }

    if (this.classInput && this.classInput.element) {
      const input = this.classInput.element.querySelector("input");
      if (input) {
        input.value = "";
      }
    }
  }
}
