<?php

declare(strict_types=1);

namespace Drupal\Tests\editor_advanced_table\FunctionalJavascript;

use Drupal\editor\Entity\Editor;
use Drupal\filter\Entity\FilterFormat;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests TableAdvanced in the editor.
 *
 * @group editor_advanced_table
 */
class TableAdvancedEditorTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ckeditor5',
    'editor',
    'editor_advanced_table',
    'filter',
    'node',
    'user',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create filter format and editor.
    $filter_format = FilterFormat::create([
      'format' => 'test_format',
      'name' => 'Test Format',
      'filters' => [],
    ]);
    $filter_format->save();

    $editor = Editor::create([
      'format' => 'test_format',
      'editor' => 'ckeditor5',
      'settings' => [
        'toolbar' => [
          'items' => ['insertTable', 'tableAdvanced'],
        ],
        'plugins' => [
          'ckeditor5_table' => [],
          'editor_advanced_table_table_advanced' => [
            'allowed_classes' => "table\ntable-striped",
            'allow_custom_class' => FALSE,
            'allow_id' => TRUE,
            'allow_language_direction' => TRUE,
          ],
        ],
      ],
    ]);
    $editor->save();

    // Create user with permission.
    $account = $this->drupalCreateUser([
      'use text format test_format',
    ]);
    $this->drupalLogin($account);
  }

  /**
   * Tests that the plugin is properly registered.
   *
   * Note: Full browser testing with CKEditor initialization requires
   * Selenium/ChromeDriver setup which is not available in all environments.
   */
  public function testPluginIsRegistered(): void {
    $manager = \Drupal::service('plugin.manager.ckeditor5.plugin');

    // Verify the plugin exists.
    $this->assertTrue($manager->hasDefinition('editor_advanced_table_table_advanced'));

    // Verify the plugin is loadable.
    $plugin = $manager->createInstance('editor_advanced_table_table_advanced');
    $this->assertInstanceOf('Drupal\editor_advanced_table\Plugin\CKEditor5Plugin\TableAdvanced', $plugin);
  }

  /**
   * Tests plugin configuration is loaded.
   */
  public function testPluginConfigurationLoaded(): void {
    $manager = \Drupal::service('plugin.manager.ckeditor5.plugin');
    $plugin_definition = $manager->getDefinition('editor_advanced_table_table_advanced');

    $this->assertNotNull($plugin_definition);
    $this->assertEquals('Drupal\editor_advanced_table\Plugin\CKEditor5Plugin\TableAdvanced', $plugin_definition->getClass());
  }

  /**
   * Tests editor settings contain plugin configuration.
   */
  public function testEditorSettingsContainPluginConfig(): void {
    $editor = Editor::load('test_format');
    $settings = $editor->getSettings();

    $this->assertArrayHasKey('plugins', $settings);
    $this->assertArrayHasKey('editor_advanced_table_table_advanced', $settings['plugins']);
  }

}
